// Verbose terminate_handler -*- C++ -*-

/*
 * Changes for traceback support:
 *
 * Copyright (C) 2009, 2010 Florian Weimer <fw@deneb.enyo.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

// The original FSF copyright header follows.
//
// Copyright (C) 2001, 2002, 2004, 2005 Free Software Foundation
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING.  If not, write to the Free
// Software Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
// USA.

// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.

#include <diagnostics/extensions/stacktrace/handler.hpp>
#include <diagnostics/extensions/stacktrace/frame_visitor.hpp>

#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <exception>
#include <stdexcept>
#include <cxxabi.h>

DIAGNOSTICS_NAMESPACE_BEGIN;
STACKTRACE_NAMESAPCE_BEGIN;

void
Traceback::InstallHandler()
{
  std::set_terminate(Traceback::TerminateHandler);
}

namespace {
  struct my_visitor
    : Traceback::FrameVisitor, Traceback::SourceVisitor {
    bool source_info;
    bool cxa_seen;

    my_visitor()
      : source_info(true), cxa_seen(false)
    {
    }

    bool Frame(void *original, const char *dso, void *resolved)
    {
      if (resolved) {
        if (!(source_info && WalkSource(dso, resolved))) {
          fprintf(stderr, "  at %s@%p\n", dso, resolved);
        }
      } else {
        fprintf(stderr, "  at <unknown>@%p\n", original);
      }
      return true;
    }

    void Frame(const char *func, const char *source, unsigned line)
    {
      if (!cxa_seen) {
        if (strcmp(func, "__cxa_throw") == 0) {
          cxa_seen = true;
        }
      } else {
        fprintf(stderr, " at %s (%s:%u)\n", func, source, line);
      }
    }
  };
}

// A replacement for the standard terminate_handler which prints
// more information about the terminating exception (if any) on
// stderr.
void
Traceback::TerminateHandler()
{
  static bool terminating;
  if (terminating) {
    fputs("terminate called recursively\n", stderr);
    abort ();
  }
  terminating = true;

  my_visitor v;

  // Make sure there was an exception; terminate is also called for an
  // attempt to rethrow when there is no suitable exception.
  std::type_info *t = abi::__cxa_current_exception_type();
  if (t) {
    // Note that "name" is the mangled name.
    char const *name = t->name();
    {
      int status = -1;
      char *dem = 0;

      dem = abi::__cxa_demangle(name, 0, 0, &status);

      fputs("Exception ", stderr);
      if (status == 0) {
        fputs(dem, stderr);
      } else {
        fputs(name, stderr);
      }

      if (status == 0) {
        free(dem);
      }
    }

    // If the exception is derived from std::exception, we can
    // give more information.
    try { throw; }
    catch (std::bad_alloc &exc) {
      char const *w = exc.what();
      fputs(": ", stderr);
      fputs(w, stderr);
      v.source_info = false; // reduce memory pressure
    } catch (std::exception &exc) {
      char const *w = exc.what();
      fputs(": ", stderr);
      fputs(w, stderr);
    } catch (...) { }
    fputc('\n', stderr);
  } else {
    fputs("Exception: std::terminate called\n", stderr);
    v.cxa_seen = true;
  }

  v.Walk();
  abort();
}

STACKTRACE_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

