/** @file
  UEFI Service Binding Protocol is defined in UEFI specification.

  The file defines the generic Service Binding Protocol functions.
  It provides services that are required to create and destroy child
  handles that support a given set of protocols.

  Copyright (c) 2006 - 2018, Intel Corporation. All rights reserved.<BR>
  SPDX-License-Identifier: BSD-2-Clause-Patent

**/

#ifndef __EFI_SERVICE_BINDING_H__
#define __EFI_SERVICE_BINDING_H__

FILE_LICENCE ( BSD2_PATENT );

///
/// Forward reference for pure ANSI compatability
///
typedef struct _EFI_SERVICE_BINDING_PROTOCOL EFI_SERVICE_BINDING_PROTOCOL;

/**
  Creates a child handle and installs a protocol.

  The CreateChild() function installs a protocol on ChildHandle.
  If ChildHandle is a pointer to NULL, then a new handle is created and returned in ChildHandle.
  If ChildHandle is not a pointer to NULL, then the protocol installs on the existing ChildHandle.

  @param  This        Pointer to the EFI_SERVICE_BINDING_PROTOCOL instance.
  @param  ChildHandle Pointer to the handle of the child to create. If it is NULL,
                      then a new handle is created. If it is a pointer to an existing UEFI handle,
                      then the protocol is added to the existing UEFI handle.

  @retval EFI_SUCCES            The protocol was added to ChildHandle.
  @retval EFI_INVALID_PARAMETER ChildHandle is NULL.
  @retval EFI_OUT_OF_RESOURCES  There are not enough resources available to create
                                the child
  @retval other                 The child handle was not created

**/
typedef
EFI_STATUS
(EFIAPI *EFI_SERVICE_BINDING_CREATE_CHILD)(
  IN     EFI_SERVICE_BINDING_PROTOCOL  *This,
  IN OUT EFI_HANDLE                    *ChildHandle
  );

/**
  Destroys a child handle with a protocol installed on it.

  The DestroyChild() function does the opposite of CreateChild(). It removes a protocol
  that was installed by CreateChild() from ChildHandle. If the removed protocol is the
  last protocol on ChildHandle, then ChildHandle is destroyed.

  @param  This        Pointer to the EFI_SERVICE_BINDING_PROTOCOL instance.
  @param  ChildHandle Handle of the child to destroy

  @retval EFI_SUCCES            The protocol was removed from ChildHandle.
  @retval EFI_UNSUPPORTED       ChildHandle does not support the protocol that is being removed.
  @retval EFI_INVALID_PARAMETER Child handle is NULL.
  @retval EFI_ACCESS_DENIED     The protocol could not be removed from the ChildHandle
                                because its services are being used.
  @retval other                 The child handle was not destroyed

**/
typedef
EFI_STATUS
(EFIAPI *EFI_SERVICE_BINDING_DESTROY_CHILD)(
  IN EFI_SERVICE_BINDING_PROTOCOL          *This,
  IN EFI_HANDLE                            ChildHandle
  );

///
/// The EFI_SERVICE_BINDING_PROTOCOL provides member functions to create and destroy
/// child handles. A driver is responsible for adding protocols to the child handle
/// in CreateChild() and removing protocols in DestroyChild(). It is also required
/// that the CreateChild() function opens the parent protocol BY_CHILD_CONTROLLER
/// to establish the parent-child relationship, and closes the protocol in DestroyChild().
/// The pseudo code for CreateChild() and DestroyChild() is provided to specify the
/// required behavior, not to specify the required implementation. Each consumer of
/// a software protocol is responsible for calling CreateChild() when it requires the
/// protocol and calling DestroyChild() when it is finished with that protocol.
///
struct _EFI_SERVICE_BINDING_PROTOCOL {
  EFI_SERVICE_BINDING_CREATE_CHILD     CreateChild;
  EFI_SERVICE_BINDING_DESTROY_CHILD    DestroyChild;
};

#endif
