package Net::Domain::TLD;
use strict;
use base qw(Exporter);
use 5.006;
our @EXPORT_OK = qw(tlds tld_exists %tld_profile);
our $VERSION = '1.73';

use warnings;
use Carp;
use Storable qw ( dclone );

use constant TLD_TYPES => qw ( new_open new_restricted gtld_open gtld_restricted gtld_new cc ccidn );

=head1 NAME

  Net::Domain::TLD - Work with TLD names 

=head1 SYNOPSIS

  use Net::Domain::TLD qw(tlds tld_exists);
  my @ccTLDs = tlds('cc');
  print "TLD ok\n" if tld_exists('ac','cc');

=head1 DESCRIPTION

  The purpose of this module is to provide user with current list of 
  available top level domain names including new ICANN additions and ccTLDs
  Currently TLD definitions have been acquired from the following sources:

  http://www.icann.org/tlds/
  http://www.dnso.org/constituency/gtld/gtld.html
  http://www.iana.org/cctld/cctld-whois.htm
  https://www.iana.org/domains/root/db

=cut

my %tld_profile = (
  reserved => {
    test => q{DNS testing names},
    example => q{Documentation names},
    invalid => q{Invalid names},
    localhost => q{Loopback names}
  },
  new_open => {
    info => q{Unrestricted use},
    xxx => q{sponsored top-level domain}
  },
  new_restricted => { 
    aero => q{Air-transport industry},
    asia => q{Companies, organisations and individuals in the Asia-Pacific region},
    arpa => q{Address and Routing Parameter Area},
    biz => q{Businesses},
    cat => q{Catalan linguistic and cultural community},
    coop => q{Cooperatives},
    jobs => q{Human Resource Management},
    mobi => q{Mobile},
    museum => q{Museums},
    name => q{For registration by individuals},
    post => q{Universal Postal Union},
    pro => q{Accountants, lawyers, and physicians},
    travel => q{Travel industry},
    tel => q{For businesses and individuals to publish contact data}
  },
  gtld_open => {
    com => q{Commercial organization},
    net => q{Network connection services provider},
    org => q{Non-profit organizations and industry standard groups}
  },
  gtld_restricted => {
    gov => q{United States Government},
    mil => q{United States Military},
    edu => q{Educational institution},
    int => q{International treaties/databases},
  },
  cc => {
    ac => q{Ascension Island},
    ad => q{Andorra},
    ae => q{United Arab Emirates},
    af => q{Afghanistan},
    ag => q{Antigua and Barbuda},
    ai => q{Anguilla},
    al => q{Albania},
    am => q{Armenia},
    an => q{Netherlands Antilles},
    ao => q{Angola},
    aq => q{Antartica},
    ar => q{Argentina},
    as => q{American Samoa},
    at => q{Austria},
    au => q{Australia},
    aw => q{Aruba},
    ax => q(Aland Islands),
    az => q{Azerbaijan},
    ba => q{Bosnia and Herzegovina},
    bb => q{Barbados},
    bd => q{Bangladesh},
    be => q{Belgium},
    bf => q{Burkina Faso},
    bg => q{Bulgaria},
    bh => q{Bahrain},
    bi => q{Burundi},
    bj => q{Benin},
    bl => q(Saint Barthelemy),
    bm => q{Bermuda},
    bn => q{Brunei Darussalam},
    bo => q{Bolivia},
    bq => q{Not assigned},
    br => q{Brazil},
    bs => q{Bahamas},
    bt => q{Bhutan},
    bv => q{Bouvet Island},
    bw => q{Botswana},
    by => q{Belarus},
    bz => q{Belize},
    ca => q{Canada},
    cc => q{Cocos (Keeling) Islands},
    cd => q{Congo, Democratic Republic of the},
    cf => q{Central African Republic},
    cg => q{Congo, Republic of},
    ch => q{Switzerland},
    ci => q{Cote d'Ivoire},
    ck => q{Cook Islands},
    cl => q{Chile},
    cm => q{Cameroon},
    cn => q{China},
    co => q{Colombia},
    cr => q{Costa Rica},
    cu => q{Cuba},
    cv => q{Cap Verde},
    cw => q{University of the Netherlands Antilles},
    cx => q{Christmas Island},
    cy => q{Cyprus},
    cz => q{Czech Republic},
    de => q{Germany},
    dj => q{Djibouti},
    dk => q{Denmark},
    dm => q{Dominica},
    do => q{Dominican Republic},
    dz => q{Algeria},
    ec => q{Ecuador},
    ee => q{Estonia},
    eg => q{Egypt},
    eh => q{Western Sahara},
    er => q{Eritrea},
    es => q{Spain},
    et => q{Ethiopia},
    eu => q{European Union},
    fi => q{Finland},
    fj => q{Fiji},
    fk => q{Falkland Islands (Malvina)},
    fm => q{Micronesia, Federal State of},
    fo => q{Faroe Islands},
    fr => q{France},
    ga => q{Gabon},
    gb => q{United Kingdom},
    gd => q{Grenada},
    ge => q{Georgia},
    gf => q{French Guiana},
    gg => q{Guernsey},
    gh => q{Ghana},
    gi => q{Gibraltar},
    gl => q{Greenland},
    gm => q{Gambia},
    gn => q{Guinea},
    gp => q{Guadeloupe},
    gq => q{Equatorial Guinea},
    gr => q{Greece},
    gs => q{South Georgia and the South Sandwich Islands},
    gt => q{Guatemala},
    gu => q{Guam},
    gw => q{Guinea-Bissau},
    gy => q{Guyana},
    hk => q{Hong Kong},
    hm => q{Heard and McDonald Islands},
    hn => q{Honduras},
    hr => q{Croatia/Hrvatska},
    ht => q{Haiti},
    hu => q{Hungary},
    id => q{Indonesia},
    ie => q{Ireland},
    il => q{Israel},
    im => q{Isle of Man},
    in => q{India},
    io => q{British Indian Ocean Territory},
    iq => q{Iraq},
    ir => q{Iran (Islamic Republic of)},
    is => q{Iceland},
    it => q{Italy},
    je => q{Jersey},
    jm => q{Jamaica},
    jo => q{Jordan},
    jp => q{Japan},
    ke => q{Kenya},
    kg => q{Kyrgyzstan},
    kh => q{Cambodia},
    ki => q{Kiribati},
    km => q{Comoros},
    kn => q{Saint Kitts and Nevis},
    kp => q{Korea, Democratic People's Republic},
    kr => q{Korea, Republic of},
    kw => q{Kuwait},
    ky => q{Cayman Islands},
    kz => q{Kazakhstan},
    la => q{Lao People's Democratic Republic},
    lb => q{Lebanon},
    lc => q{Saint Lucia},
    li => q{Liechtenstein},
    lk => q{Sri Lanka},
    lr => q{Liberia},
    ls => q{Lesotho},
    lt => q{Lithuania},
    lu => q{Luxembourg},
    lv => q{Latvia},
    ly => q{Libyan Arab Jamahiriya},
    ma => q{Morocco},
    mc => q{Monaco},
    md => q{Moldova, Republic of},
    me => q(Montenegro),
    mf => q{Saint Martin (French part)},
    mg => q{Madagascar},
    mh => q{Marshall Islands},
    mk => q{Macedonia, Former Yugoslav Republic},
    ml => q{Mali},
    mm => q{Myanmar},
    mn => q{Mongolia},
    mo => q{Macau},
    mp => q{Northern Mariana Islands},
    mq => q{Martinique},
    mr => q{Mauritania},
    ms => q{Montserrat},
    mt => q{Malta},
    mu => q{Mauritius},
    mv => q{Maldives},
    mw => q{Malawi},
    mx => q{Mexico},
    my => q{Malaysia},
    mz => q{Mozambique},
    na => q{Namibia},
    nc => q{New Caledonia},
    ne => q{Niger},
    nf => q{Norfolk Island},
    ng => q{Nigeria},
    ni => q{Nicaragua},
    nl => q{Netherlands},
    no => q{Norway},
    np => q{Nepal},
    nr => q{Nauru},
    nu => q{Niue},
    nz => q{New Zealand},
    om => q{Oman},
    pa => q{Panama},
    pe => q{Peru},
    pf => q{French Polynesia},
    pg => q{Papua New Guinea},
    ph => q{Philippines},
    pk => q{Pakistan},
    pl => q{Poland},
    pm => q{St. Pierre and Miquelon},
    pn => q{Pitcairn Island},
    pr => q{Puerto Rico},
    ps => q{Palestinian Territories},
    pt => q{Portugal},
    pw => q{Palau},
    py => q{Paraguay},
    qa => q{Qatar},
    re => q{Reunion Island},
    ro => q{Romania},
    rs => q(Serbia),
    ru => q{Russian Federation},
    rw => q{Rwanda},
    sa => q{Saudi Arabia},
    sb => q{Solomon Islands},
    sc => q{Seychelles},
    sd => q{Sudan},
    se => q{Sweden},
    sg => q{Singapore},
    sh => q{St. Helena},
    si => q{Slovenia},
    sj => q{Svalbard and Jan Mayen Islands},
    sk => q{Slovak Republic},
    sl => q{Sierra Leone},
    sm => q{San Marino},
    sn => q{Senegal},
    so => q{Somalia},
    sr => q{Suriname},
    ss => q{Not assigned},
    st => q{Sao Tome and Principe},
    su => q{Soviet Union},
    sv => q{El Salvador},
    sx => q{SX Registry SA B.V.},
    sy => q{Syrian Arab Republic},
    sz => q{Swaziland},
    tc => q{Turks and Caicos Islands},
    td => q{Chad},
    tf => q{French Southern Territories},
    tg => q{Togo},
    th => q{Thailand},
    tj => q{Tajikistan},
    tk => q{Tokelau},
    tl => q{Timor-Leste},
    tm => q{Turkmenistan},
    tn => q{Tunisia},
    to => q{Tonga},
    tp => q{East Timor},
    tr => q{Turkey},
    tt => q{Trinidad and Tobago},
    tv => q{Tuvalu},
    tw => q{Taiwan},
    tz => q{Tanzania},
    ua => q{Ukraine},
    ug => q{Uganda},
    uk => q{United Kingdom},
    um => q{US Minor Outlying Islands},
    us => q{United States},
    uy => q{Uruguay},
    uz => q{Uzbekistan},
    va => q{Holy See (City Vatican State)},
    vc => q{Saint Vincent and the Grenadines},
    ve => q{Venezuela},
    vg => q{Virgin Islands (British)},
    vi => q{Virgin Islands (USA)},
    vn => q{Vietnam},
    vu => q{Vanuatu},
    wf => q{Wallis and Futuna Islands},
    ws => q{Western Samoa},
    ye => q{Yemen},
    yt => q{Mayotte},
    yu => q{Yugoslavia},
    za => q{South Africa},
    zm => q{Zambia},
    zw => q{Zimbabwe}
  },
  ccidn => {
    'xn--0zwm56d' => q{Internet Assigned Numbers Authority},
    'xn--11b5bs3a9aj6g' => q{Internet Assigned Numbers Authority},
    'xn--1qqw23a' => q{Guangzhou YU Wei Information Technology Co., Ltd.},
    'xn--3bst00m' => q{Eagle Horizon Limited},
    'xn--3ds443g' => q{TLD REGISTRY LIMITED},
    'xn--3e0b707e' => q{KISA (Korea Internet & Security Agency)},
    'xn--45brj9c' => q{National Internet Exchange of India},
    'xn--45q11c' => q{Zodiac Scorpio Limited},
    'xn--4gbrim' => q{Suhub Electronic Establishment},
    'xn--54b7fta0cc' => q{Not assigned},
    'xn--55qw42g' => q{China Organizational Name Administration Center},
    'xn--55qx5d' => q{Computer Network Information Center of Chinese Academy of Sciences （China Internet Network Information Center）},
    'xn--6frz82g' => q{Afilias Limited},
    'xn--6qq986b3xl' => q{Tycoon Treasure Limited},
    'xn--80adxhks' => q{Foundation for Assistance for Internet Technologies and Infrastructure Development (FAITID)},
    'xn--80akhbyknj4f' => q{Internet Assigned Numbers Authority},
    'xn--80ao21a' => q{Association of IT Companies of Kazakhstan},
    'xn--80asehdb' => q{CORE Association},
    'xn--80aswg' => q{CORE Association},
    'xn--90a3ac' => q{Serbian National Internet Domain Registry (RNIDS)},
    'xn--90ais' => q{Not assigned},
    'xn--9t4b11yi5a' => q{Internet Assigned Numbers Authority},
    'xn--b4w605ferd' => q{Temasek Holdings (Private) Limited},
    'xn--c1avg' => q{Public Interest Registry},
    'xn--cg4bki' => q{SAMSUNG SDS CO., LTD},
    'xn--clchc0ea0b2g2a9gcd' => q{Singapore Network Information Centre (SGNIC) Pte Ltd},
    'xn--czr694b' => q{HU YI GLOBAL INFORMATION RESOURCES(HOLDING) COMPANY.HONGKONG LIMITED},
    'xn--czrs0t' => q{Wild Island, LLC},
    'xn--czru2d' => q{Zodiac Aquarius Limited},
    'xn--d1acj3b' => q{The Foundation for Network Initiatives “The Smart Internet”},
    'xn--d1alf' => q{Macedonian Academic Research Network Skopje},
    'xn--deba0ad' => q{Internet Assigned Numbers Authority},
    'xn--fiq228c5hs' => q{TLD REGISTRY LIMITED},
    'xn--fiq64b' => q{CITIC Group Corporation},
    'xn--fiqs8s' => q{China Internet Network Information Center},
    'xn--fiqz9s' => q{China Internet Network Information Center},
    'xn--flw351e' => q{Charleston Road Registry Inc.},
    'xn--fpcrj9c3d' => q{National Internet Exchange of India},
    'xn--fzc2c9e2c' => q{LK Domain Registry},
    'xn--g6w251d' => q{Internet Assigned Numbers Authority},
    'xn--gecrj9c' => q{National Internet Exchange of India},
    'xn--h2brj9c' => q{National Internet Exchange of India},
    'xn--hgbk6aj7f53bba' => q{Internet Assigned Numbers Authority},
    'xn--hlcj6aya9esc7a' => q{Internet Assigned Numbers Authority},
    'xn--hxt814e' => q{Zodiac Libra Limited},
    'xn--i1b6b1a6a2e' => q{Public Interest Registry},
    'xn--io0a7i' => q{Computer Network Information Center of Chinese Academy of Sciences （China Internet Network Information Center）},
    'xn--j1amh' => q{Ukrainian Network Information Centre (UANIC), Inc.},
    'xn--j6w193g' => q{Hong Kong Internet Registration Corporation Ltd.},
    'xn--jxalpdlp' => q{Internet Assigned Numbers Authority},
    'xn--kgbechtv' => q{Internet Assigned Numbers Authority},
    'xn--kprw13d' => q{Taiwan Network Information Center (TWNIC)},
    'xn--kpry57d' => q{Taiwan Network Information Center (TWNIC)},
    'xn--kput3i' => q{Beijing RITT-Net Technology Development Co., Ltd},
    'xn--l1acc' => q{Datacom Co.,Ltd},
    'xn--lgbbat1ad8j' => q{CERIST},
    'xn--mgb9awbf' => q{Telecommunications Regulatory Authority (TRA)},
    'xn--mgba3a4f16a' => q{Institute for Research in Fundamental Sciences (IPM)},
    'xn--mgbaam7a8h' => q{Telecommunications Regulatory Authority (TRA)},
    'xn--mgbab2bd' => q{CORE Association},
    'xn--mgbai9azgqp6j' => q{Not assigned},
    'xn--mgbayh7gpa' => q{National Information Technology Center (NITC)},
    'xn--mgbbh1a71e' => q{National Internet Exchange of India},
    'xn--mgbc0a9azcg' => q{Agence Nationale de Rglementation des Tlcommunications (ANRT)},
    'xn--mgberp4a5d4ar' => q{Communications and Information Technology Commission},
    'xn--mgbpl2fh' => q{Not assigned},
    'xn--mgbtx2b' => q{Not assigned},
    'xn--mgbx4cd0ab' => q{MYNIC Berhad},
    'xn--ngbc5azd' => q{International Domain Registry Pty. Ltd.},
    'xn--node' => q{Information Technologies Development Center (ITDC)},
    'xn--nqv7f' => q{Public Interest Registry},
    'xn--nqv7fs00ema' => q{Public Interest Registry},
    'xn--o3cw4h' => q{Thai Network Information Center Foundation},
    'xn--ogbpf8fl' => q{National Agency for Network Services (NANS)},
    'xn--p1acf' => q{Rusnames Limited},
    'xn--p1ai' => q{Coordination Center for TLD RU},
    'xn--pgbs0dh' => q{Agence Tunisienne d'Internet},
    'xn--q9jyb4c' => q{Charleston Road Registry Inc.},
    'xn--qcka1pmc' => q{Charleston Road Registry Inc.},
    'xn--rhqv96g' => q{Stable Tone Limited},
    'xn--s9brj9c' => q{National Internet Exchange of India},
    'xn--ses554g' => q{KNET Co., Ltd},
    'xn--unup4y' => q{Spring Fields, LLC},
    'xn--vermgensberater-ctb' => q{Deutsche Vermgensberatung Aktiengesellschaft DVAG},
    'xn--vermgensberatung-pwb' => q{Deutsche Vermgensberatung Aktiengesellschaft DVAG},
    'xn--vhquv' => q{Dash McCook, LLC},
    'xn--wgbh1c' => q{National Telecommunication Regulatory Authority - NTRA},
    'xn--wgbl6a' => q{Communications Regulatory Authority},
    'xn--xhq521b' => q{Guangzhou YU Wei Information Technology Co., Ltd.},
    'xn--xkc2al3hye2a' => q{LK Domain Registry},
    'xn--xkc2dl3a5ee0h' => q{National Internet Exchange of India},
    'xn--y9a3aq' => q{Not assigned},
    'xn--yfro4i67o' => q{Singapore Network Information Centre (SGNIC) Pte Ltd},
    'xn--ygbi2ammx' => q{Ministry of Telecom & Information Technology (MTIT)},
    'xn--zckzah' => q{Internet Assigned Numbers Authority},
    'xn--zfr164b' => q{China Organizational Name Administration Center},
  },
  gtld_new => {
    'abogado' => q{Top Level Domain Holdings Limited},
    'ac' => q{Network Information Center (AC Domain Registry) c/o Cable and Wireless (Ascension Island)},
    'academy' => q{Half Oaks, LLC},
    'accountants' => q{Knob Town, LLC},
    'active' => q{The Active Network, Inc},
    'actor' => q{United TLD Holdco Ltd.},
    'ad' => q{Andorra Telecom},
    'adult' => q{ICM Registry AD LLC},
    'ae' => q{Telecommunication Regulatory Authority (TRA)},
    'aero' => q{Societe Internationale de Telecommunications Aeronautique (SITA INC USA)},
    'af' => q{Ministry of Communications and IT},
    'ag' => q{UHSA School of Medicine},
    'agency' => q{Steel Falls, LLC},
    'ai' => q{Government of Anguilla},
    'airforce' => q{United TLD Holdco Ltd.},
    'al' => q{Electronic and Postal Communications Authority - AKEP},
    'allfinanz' => q{Allfinanz Deutsche Vermgensberatung Aktiengesellschaft},
    'alsace' => q{REGION D ALSACE},
    'am' => q{Internet Society},
    'amsterdam' => q{Gemeente Amsterdam},
    'an' => q{University of Curacao},
    'android' => q{Charleston Road Registry Inc.},
    'ao' => q{Faculdade de Engenharia da Universidade Agostinho Neto},
    'aq' => q{Antarctica Network Information Centre Limited},
    'aquarelle' => q{Aquarelle.com},
    'ar' => q{Presidencia de la Nación – Secretaría Legal y Técnica},
    'archi' => q{STARTING DOT LIMITED},
    'army' => q{United TLD Holdco Ltd.},
    'arpa' => q{Internet Architecture Board (IAB)},
    'as' => q{AS Domain Registry},
    'asia' => q{DotAsia Organisation Ltd.},
    'associates' => q{Baxter Hill, LLC},
    'at' => q{nic.at GmbH},
    'attorney' => q{United TLD Holdco, Ltd},
    'au' => q{.au Domain Administration (auDA)},
    'auction' => q{United TLD HoldCo, Ltd.},
    'audio' => q{Uniregistry, Corp.},
    'autos' => q{DERAutos, LLC},
    'aw' => q{SETAR},
    'ax' => q{lands landskapsregering},
    'axa' => q{AXA SA},
    'az' => q{IntraNS},
    'ba' => q{Universtiy Telinformatic Centre (UTIC)},
    'band' => q{United TLD Holdco, Ltd},
    'bank' => q{fTLD Registry Services, LLC},
    'bar' => q{Punto 2012 Sociedad Anonima Promotora de Inversion de Capital Variable},
    'barclaycard' => q{Barclays Bank PLC},
    'barclays' => q{Barclays Bank PLC},
    'bargains' => q{Half Hallow, LLC},
    'bayern' => q{Bayern Connect GmbH},
    'bb' => q{Government of Barbados Ministry of Economic Affairs and Development Telecommunications Unit},
    'bd' => q{Ministry of Post & Telecommunications Bangladesh Secretariat},
    'be' => q{DNS Belgium vzw/asbl},
    'beer' => q{Top Level Domain Holdings Limited},
    'berlin' => q{dotBERLIN GmbH & Co. KG},
    'best' => q{BestTLD Pty Ltd},
    'bf' => q{ARCE-Autorit de Rgulation des Communications Electroniques},
    'bg' => q{Register.BG},
    'bh' => q{Telecommunications Regulatory Authority (TRA)},
    'bi' => q{Centre National de l'Informatique},
    'bid' => q{dot Bid Limited},
    'bike' => q{Grand Hollow, LLC},
    'bio' => q{STARTING DOT LIMITED},
    'biz' => q{NeuStar, Inc.},
    'bj' => q{Benin Telecoms S.A.},
    'bl' => q{Not assigned},
    'black' => q{Afilias Limited},
    'blackfriday' => q{Uniregistry, Corp.},
    'bloomberg' => q{Bloomberg IP Holdings LLC},
    'blue' => q{Afilias Limited},
    'bm' => q{Registry General Ministry of Labour and Immigration},
    'bmw' => q{Bayerische Motoren Werke Aktiengesellschaft},
    'bn' => q{Telekom Brunei Berhad},
    'bnpparibas' => q{BNP Paribas},
    'bo' => q{Agencia para el Desarrollo de la Informacin de la Sociedad en Bolivia},
    'boo' => q{Charleston Road Registry Inc.},
    'boutique' => q{Over Galley, LLC},
    'bq' => q{Not assigned},
    'br' => q{Comite Gestor da Internet no Brasil},
    'brussels' => q{DNS.be vzw},
    'bs' => q{The College of the Bahamas},
    'bt' => q{Ministry of Information and Communications},
    'budapest' => q{Top Level Domain Holdings Limited},
    'build' => q{Plan Bee LLC},
    'builders' => q{Atomic Madison, LLC},
    'business' => q{Spring Cross, LLC},
    'buzz' => q{DOTSTRATEGY CO.},
    'bv' => q{UNINETT Norid A/S},
    'bw' => q{Botswana Communications Regulatory Authority (BOCRA)},
    'by' => q{Reliable Software Inc.},
    'bz' => q{University of Belize},
    'bzh' => q{Association www.bzh},
    'ca' => q{Canadian Internet Registration Authority (CIRA) Autorite Canadienne pour les Enregistrements Internet (ACEI)},
    'cab' => q{Half Sunset, LLC},
    'cal' => q{Charleston Road Registry Inc.},
    'camera' => q{Atomic Maple, LLC},
    'camp' => q{Delta Dynamite, LLC},
    'cancerresearch' => q{Australian Cancer Research Foundation},
    'capetown' => q{ZA Central Registry NPC trading as ZA Central Registry},
    'capital' => q{Delta Mill, LLC},
    'caravan' => q{Caravan International, Inc.},
    'cards' => q{Foggy Hollow, LLC},
    'care' => q{Goose Cross, LLC},
    'career' => q{dotCareer LLC},
    'careers' => q{Wild Corner, LLC},
    'cartier' => q{Richemont DNS Inc.},
    'casa' => q{Top Level Domain Holdings Limited},
    'cash' => q{Delta Lake, LLC},
    'cat' => q{Fundacio puntCAT},
    'catering' => q{New Falls. LLC},
    'cc' => q{eNIC Cocos (Keeling) Islands Pty. Ltd. d/b/a Island Internet Services},
    'cd' => q{Office Congolais des Postes et Tlcommunications - OCPT},
    'center' => q{Tin Mill, LLC},
    'ceo' => q{CEOTLD Pty Ltd},
    'cern' => q{European Organization for Nuclear Research ("CERN")},
    'cf' => q{Societe Centrafricaine de Telecommunications (SOCATEL)},
    'cg' => q{ONPT Congo and Interpoint Switzerland},
    'ch' => q{SWITCH The Swiss Education & Research Network},
    'channel' => q{Charleston Road Registry Inc.},
    'cheap' => q{Sand Cover, LLC},
    'christmas' => q{Uniregistry, Corp.},
    'chrome' => q{Charleston Road Registry Inc.},
    'church' => q{Holly Fileds, LLC},
    'ci' => q{INP-HB Institut National Polytechnique Felix Houphouet Boigny},
    'citic' => q{CITIC Group Corporation},
    'city' => q{Snow Sky, LLC},
    'ck' => q{Telecom Cook Islands Ltd.},
    'cl' => q{NIC Chile (University of Chile)},
    'claims' => q{Black Corner, LLC},
    'cleaning' => q{Fox Shadow, LLC},
    'click' => q{Uniregistry, Corp.},
    'clinic' => q{Goose Park, LLC},
    'clothing' => q{Steel Lake, LLC},
    'club' => q{.CLUB DOMAINS, LLC},
    'cm' => q{Cameroon Telecommunications (CAMTEL)},
    'cn' => q{Computer Network Information Center, Chinese Academy of Sciences},
    'co' => q{.CO Internet S.A.S.},
    'coach' => q{Koko Island, LLC},
    'codes' => q{Puff Willow, LLC},
    'coffee' => q{Trixy Cover, LLC},
    'college' => q{XYZ.COM LLC},
    'cologne' => q{NetCologne Gesellschaft fr Telekommunikation mbH},
    'com' => q{VeriSign Global Registry Services},
    'community' => q{Fox Orchard, LLC},
    'company' => q{Silver Avenue, LLC},
    'computer' => q{Pine Mill, LLC},
    'condos' => q{Pine House, LLC},
    'construction' => q{Fox Dynamite, LLC},
    'consulting' => q{United TLD Holdco, LTD.},
    'contractors' => q{Magic Woods, LLC},
    'cooking' => q{Top Level Domain Holdings Limited},
    'cool' => q{Koko Lake, LLC},
    'coop' => q{DotCooperation LLC},
    'country' => q{Top Level Domain Holdings Limited},
    'cr' => q{National Academy of Sciences Academia Nacional de Ciencias},
    'credit' => q{Snow Shadow, LLC},
    'creditcard' => q{Binky Frostbite, LLC},
    'cricket' => q{dot Cricket Limited},
    'crs' => q{Federated Co-operatives Limited},
    'cruises' => q{Spring Way, LLC},
    'cu' => q{CENIAInternet Industria y San Jose Capitolio Nacional},
    'cuisinella' => q{SALM S.A.S.},
    'cv' => q{Agncia Nacional das Comunicaes (ANAC)},
    'cw' => q{University of Curacao},
    'cx' => q{Christmas Island Internet Administration Limited},
    'cy' => q{University of Cyprus},
    'cymru' => q{Nominet UK},
    'cz' => q{CZ.NIC, z.s.p.o},
    'dabur' => q{Dabur India Limited},
    'dad' => q{Charleston Road Registry Inc.},
    'dance' => q{United TLD Holdco Ltd.},
    'dating' => q{Pine Fest, LLC},
    'day' => q{Charleston Road Registry Inc.},
    'dclk' => q{Charleston Road Registry Inc.},
    'de' => q{DENIC eG},
    'deals' => q{Sand Sunset, LLC},
    'degree' => q{United TLD Holdco, Ltd},
    'delivery' => q{Steel Station, LLC},
    'democrat' => q{United TLD Holdco Ltd.},
    'dental' => q{Tin Birch, LLC},
    'dentist' => q{United TLD Holdco, Ltd},
    'desi' => q{Desi Networks LLC},
    'design' => q{Top Level Design, LLC},
    'dev' => q{Charleston Road Registry Inc.},
    'diamonds' => q{John Edge, LLC},
    'diet' => q{Uniregistry, Corp.},
    'digital' => q{Dash Park, LLC},
    'direct' => q{Half Trail, LLC},
    'directory' => q{Extra Madison, LLC},
    'discount' => q{Holly Hill, LLC},
    'dj' => q{Djibouti Telecom S.A},
    'dk' => q{Dansk Internet Forum},
    'dm' => q{DotDM Corporation},
    'dnp' => q{Dai Nippon Printing Co., Ltd.},
    'do' => q{Pontificia Universidad Catolica Madre y Maestra Recinto Santo Tomas de Aquino},
    'docs' => q{Charleston Road Registry Inc.},
    'domains' => q{Sugar Cross, LLC},
    'doosan' => q{Doosan Corporation},
    'durban' => q{ZA Central Registry NPC trading as ZA Central Registry},
    'dvag' => q{Deutsche Vermgensberatung Aktiengesellschaft DVAG},
    'dz' => q{CERIST},
    'eat' => q{Charleston Road Registry Inc.},
    'ec' => q{NIC.EC (NICEC) S.A.},
    'edu' => q{EDUCAUSE},
    'education' => q{Brice Way, LLC},
    'ee' => q{Eesti Interneti Sihtasutus (EIS)},
    'eg' => q{Egyptian Universities Network (EUN) Supreme Council of Universities},
    'eh' => q{Not assigned},
    'email' => q{Spring Madison, LLC},
    'emerck' => q{Merck KGaA},
    'energy' => q{Binky Birch, LLC},
    'engineer' => q{United TLD Holdco Ltd.},
    'engineering' => q{Romeo Canyon},
    'enterprises' => q{Snow Oaks, LLC},
    'equipment' => q{Corn Station, LLC},
    'er' => q{Eritrea Telecommunication Services Corporation (EriTel)},
    'es' => q{Red.es},
    'esq' => q{Charleston Road Registry Inc.},
    'estate' => q{Trixy Park, LLC},
    'et' => q{Ethio telecom},
    'eu' => q{EURid vzw/asbl},
    'eurovision' => q{European Broadcasting Union (EBU)},
    'eus' => q{Puntueus Fundazioa},
    'events' => q{Pioneer Maple, LLC},
    'everbank' => q{EverBank},
    'exchange' => q{Spring Falls, LLC},
    'expert' => q{Magic Pass, LLC},
    'exposed' => q{Victor Beach, LLC},
    'fail' => q{Atomic Pipe, LLC},
    'farm' => q{Just Maple, LLC},
    'fashion' => q{Top Level Domain Holdings Limited},
    'feedback' => q{Top Level Spectrum, Inc.},
    'fi' => q{Finnish Communications Regulatory Authority},
    'finance' => q{Cotton Cypress, LLC},
    'financial' => q{Just Cover, LLC},
    'firmdale' => q{Firmdale Holdings Limited},
    'fish' => q{Fox Woods, LLC},
    'fishing' => q{Top Level Domain Holdings Limited},
    'fit' => q{Minds + Machines Group Limited},
    'fitness' => q{Brice Orchard, LLC},
    'fj' => q{The University of the South Pacific IT Services},
    'fk' => q{Falkland Islands Government},
    'flights' => q{Fox Station, LLC},
    'florist' => q{Half Cypress, LLC},
    'flowers' => q{Uniregistry, Corp.},
    'flsmidth' => q{FLSmidth A/S},
    'fly' => q{Charleston Road Registry Inc.},
    'fm' => q{FSM Telecommunications Corporation},
    'fo' => q{FO Council},
    'foo' => q{Charleston Road Registry Inc.},
    'forsale' => q{United TLD Holdco, LLC},
    'foundation' => q{John Dale, LLC},
    'fr' => q{Association Franaise pour le Nommage Internet en Coopration (A.F.N.I.C.)},
    'frl' => q{FRLregistry B.V.},
    'frogans' => q{OP3FT},
    'fund' => q{John Castle, LLC},
    'furniture' => q{Lone Fields, LLC},
    'futbol' => q{United TLD Holdco, Ltd.},
    'ga' => q{Agence Nationale des Infrastructures Numriques et des Frquences (ANINF)},
    'gal' => q{Asociacin puntoGAL},
    'gallery' => q{Sugar House, LLC},
    'garden' => q{Top Level Domain Holdings Limited},
    'gb' => q{Reserved Domain - IANA},
    'gbiz' => q{Charleston Road Registry Inc.},
    'gd' => q{The National Telecommunications Regulatory Commission (NTRC)},
    'ge' => q{Caucasus Online},
    'gent' => q{COMBELL GROUP NV/SA},
    'gf' => q{Net Plus},
    'gg' => q{Island Networks Ltd.},
    'ggee' => q{GMO Internet, Inc.},
    'gh' => q{Network Computer Systems Limited},
    'gi' => q{Sapphire Networks},
    'gift' => q{Uniregistry, Corp.},
    'gifts' => q{Goose Sky, LLC},
    'gives' => q{United TLD Holdco Ltd.},
    'gl' => q{TELE Greenland A/S},
    'glass' => q{Black Cover, LLC},
    'gle' => q{Charleston Road Registry Inc.},
    'global' => q{Dot GLOBAL AS},
    'globo' => q{Globo Comunicao e Participaes S.A},
    'gm' => q{GM-NIC},
    'gmail' => q{Charleston Road Registry Inc.},
    'gmo' => q{GMO Internet, Inc.},
    'gmx' => q{1&1 Mail & Media GmbH},
    'gn' => q{Centre National des Sciences Halieutiques de Boussoura},
    'goog' => q{Charleston Road Registry Inc.},
    'google' => q{Charleston Road Registry Inc.},
    'gop' => q{Republican State Leadership Committee, Inc.},
    'gov' => q{General Services Administration Attn: QTDC, 2E08 (.gov Domain Registration)},
    'gp' => q{Networking Technologies Group},
    'gq' => q{GETESA},
    'gr' => q{ICS-FORTH GR},
    'graphics' => q{Over Madison, LLC},
    'gratis' => q{Pioneer Tigers, LLC},
    'green' => q{Afilias Limited},
    'gripe' => q{Corn Sunset, LLC},
    'gs' => q{Government of South Georgia and South Sandwich Islands (GSGSSI)},
    'gt' => q{Universidad del Valle de Guatemala},
    'gu' => q{University of Guam Computer Center},
    'guide' => q{Snow Moon, LLC},
    'guitars' => q{Uniregistry, Corp.},
    'guru' => q{Pioneer Cypress, LLC},
    'gw' => q{Autoridade Reguladora Nacional - Tecnologias de Informao e Comunicao da Guin-Bissau},
    'gy' => q{University of Guyana},
    'hamburg' => q{Hamburg Top-Level-Domain GmbH},
    'hangout' => q{Charleston Road Registry Inc.},
    'haus' => q{United TLD Holdco, LTD.},
    'healthcare' => q{Silver Glen, LLC},
    'help' => q{Uniregistry, Corp.},
    'here' => q{Charleston Road Registry Inc.},
    'hermes' => q{Hermes International},
    'hiphop' => q{Uniregistry, Corp.},
    'hiv' => q{dotHIV gemeinnuetziger e.V.},
    'hk' => q{Hong Kong Internet Registration Corporation Ltd.},
    'hm' => q{HM Domain Registry},
    'hn' => q{Red de Desarrollo Sostenible Honduras},
    'holdings' => q{John Madison, LLC},
    'holiday' => q{Goose Woods, LLC},
    'homes' => q{DERHomes, LLC},
    'horse' => q{Top Level Domain Holdings Limited},
    'host' => q{DotHost Inc.},
    'hosting' => q{Uniregistry, Corp.},
    'house' => q{Sugar Park, LLC},
    'how' => q{Charleston Road Registry Inc.},
    'hr' => q{CARNet - Croatian Academic and Research Network},
    'ht' => q{Consortium FDS/RDDH},
    'hu' => q{Council of Hungarian Internet Providers (CHIP)},
    'ibm' => q{International Business Machines Corporation},
    'id' => q{Perkumpulan Pengelola Nama Domain Internet Indonesia (PANDI)},
    'ie' => q{University College Dublin Computing Services Computer Centre},
    'ifm' => q{ifm electronic gmbh},
    'il' => q{Internet Society of Israel},
    'im' => q{Isle of Man Government},
    'immo' => q{Auburn Bloom, LLC},
    'immobilien' => q{United TLD Holdco Ltd.},
    'in' => q{National Internet Exchange of India},
    'industries' => q{Outer House, LLC},
    'info' => q{Afilias Limited},
    'ing' => q{Charleston Road Registry Inc.},
    'ink' => q{Top Level Design, LLC},
    'institute' => q{Outer Maple, LLC},
    'insure' => q{Pioneer Willow, LLC},
    'int' => q{Internet Assigned Numbers Authority},
    'international' => q{Wild Way, LLC},
    'investments' => q{Holly Glen, LLC},
    'io' => q{IO Top Level Domain Registry Cable and Wireless},
    'iq' => q{Communications and Media Commission (CMC)},
    'ir' => q{Institute for Research in Fundamental Sciences},
    'irish' => q{Dot-Irish LLC},
    'is' => q{ISNIC - Internet Iceland ltd.},
    'it' => q{IIT - CNR},
    'iwc' => q{Richemont DNS Inc.},
    'jcb' => q{JCB Co., Ltd.},
    'je' => q{Island Networks (Jersey) Ltd.},
    'jetzt' => q{New TLD Company AB},
    'jm' => q{University of West Indies},
    'jo' => q{National Information Technology Center (NITC)},
    'jobs' => q{Employ Media LLC},
    'joburg' => q{ZA Central Registry NPC trading as ZA Central Registry},
    'jp' => q{Japan Registry Services Co., Ltd.},
    'juegos' => q{Uniregistry, Corp.},
    'kaufen' => q{United TLD Holdco Ltd.},
    'kddi' => q{KDDI CORPORATION},
    'ke' => q{Kenya Network Information Center (KeNIC)},
    'kg' => q{AsiaInfo Telecommunication Enterprise},
    'kh' => q{Ministry of Post and Telecommunications},
    'ki' => q{Ministry of Communications, Transport, and Tourism Development},
    'kim' => q{Afilias Limited},
    'kitchen' => q{Just Goodbye, LLC},
    'kiwi' => q{DOT KIWI LIMITED},
    'km' => q{Comores Telecom},
    'kn' => q{Ministry of Finance, Sustainable Development Information & Technology},
    'koeln' => q{NetCologne Gesellschaft fr Telekommunikation mbH},
    'kp' => q{Star Joint Venture Company},
    'kr' => q{Korea Internet & Security Agency (KISA)},
    'krd' => q{KRG Department of Information Technology},
    'kred' => q{KredTLD Pty Ltd},
    'kw' => q{Ministry of Communications},
    'ky' => q{The Information and Communications Technology Authority},
    'kyoto' => q{Academic Institution: Kyoto Jyoho Gakuen},
    'kz' => q{Association of IT Companies of Kazakhstan},
    'la' => q{Lao National Internet Committee (LANIC), Ministry of Posts and Telecommunications},
    'lacaixa' => q{CAIXA D'ESTALVIS I PENSIONS DE BARCELONA},
    'land' => q{Pine Moon, LLC},
    'lat' => q{ECOM-LAC Federacin de Latinoamrica y el Caribe para Internet y el Comercio Electrnico},
    'latrobe' => q{La Trobe University},
    'lawyer' => q{United TLD Holdco, Ltd},
    'lb' => q{American University of Beirut Computing and Networking Services},
    'lc' => q{University of Puerto Rico},
    'lds' => q{IRI Domain Management, LLC},
    'lease' => q{Victor Trail, LLC},
    'legal' => q{Blue Falls, LLC},
    'lgbt' => q{Afilias Limited},
    'li' => q{Universitaet Liechtenstein},
    'lidl' => q{Schwarz Domains und Services GmbH & Co. KG},
    'life' => q{Trixy Oaks, LLC},
    'lighting' => q{John McCook, LLC},
    'limited' => q{Big Fest, LLC},
    'limo' => q{Hidden Frostbite, LLC},
    'link' => q{Uniregistry, Corp.},
    'lk' => q{Council for Information Technology LK Domain Registrar},
    'loans' => q{June Woods, LLC},
    'london' => q{Dot London Domains Limited},
    'lotte' => q{Lotte Holdings Co., Ltd.},
    'lotto' => q{Afilias Limited},
    'lr' => q{Data Technology Solutions, Inc.},
    'ls' => q{National University of Lesotho},
    'lt' => q{Kaunas University of Technology},
    'ltda' => q{InterNetX Corp.},
    'lu' => q{RESTENA},
    'luxe' => q{Top Level Domain Holdings Limited},
    'luxury' => q{Luxury Partners LLC},
    'lv' => q{University of Latvia Institute of Mathematics and Computer Science Department of Network Solutions (DNS)},
    'ly' => q{General Post and Telecommunication Company},
    'ma' => q{Agence Nationale de Rglementation des Tlcommunications (ANRT)},
    'madrid' => q{Comunidad de Madrid},
    'maison' => q{Victor Frostbite, LLC},
    'management' => q{John Goodbye, LLC},
    'mango' => q{PUNTO FA S.L.},
    'market' => q{Unitied TLD Holdco, Ltd},
    'marketing' => q{Fern Pass, LLC},
    'marriott' => q{Marriott Worldwide Corporation},
    'mc' => q{Gouvernement de Monaco Direction des Communications Electroniques},
    'md' => q{MoldData S.E.},
    'me' => q{Government of Montenegro},
    'media' => q{Grand Glen, LLC},
    'meet' => q{Afilias Limited},
    'melbourne' => q{The Crown in right of the State of Victoria, represented by its Department of State Development, Business and Innovation},
    'meme' => q{Charleston Road Registry Inc.},
    'memorial' => q{Dog Beach, LLC},
    'menu' => q{Wedding TLD2, LLC},
    'mf' => q{Not assigned},
    'mg' => q{NIC-MG (Network Information Center Madagascar)},
    'mh' => q{Office of the Cabinet},
    'miami' => q{Top Level Domain Holdings Limited},
    'mil' => q{DoD Network Information Center},
    'mini' => q{Bayerische Motoren Werke Aktiengesellschaft},
    'mk' => q{Macedonian Academic Research Network Skopje},
    'ml' => q{Agence des Technologies de l’Information et de la Communication},
    'mm' => q{Ministry of Communications, Posts & Telegraphs},
    'mn' => q{Datacom Co., Ltd.},
    'mo' => q{Bureau of Telecommunications Regulation (DSRT)},
    'mobi' => q{Afilias Technologies Limited dba dotMobi},
    'moda' => q{United TLD Holdco Ltd.},
    'moe' => q{Interlink Co., Ltd.},
    'monash' => q{Monash University},
    'money' => q{Outer McCook, LLC},
    'mormon' => q{IRI Domain Management, LLC ("Applicant")},
    'mortgage' => q{United TLD Holdco, Ltd},
    'moscow' => q{Foundation for Assistance for Internet Technologies and Infrastructure Development (FAITID)},
    'motorcycles' => q{DERMotorcycles, LLC},
    'mov' => q{Charleston Road Registry Inc.},
    'mp' => q{Saipan Datacom, Inc.},
    'mq' => q{MEDIASERV},
    'mr' => q{Universit des Sciences, de Technologie et de Mdecine},
    'ms' => q{MNI Networks Ltd.},
    'mt' => q{NIC (Malta)},
    'mu' => q{Internet Direct Ltd},
    'museum' => q{Museum Domain Management Association},
    'mv' => q{Dhiraagu Pvt. Ltd. (DHIVEHINET)},
    'mw' => q{Malawi Sustainable Development Network Programme (Malawi SDNP)},
    'mx' => q{NIC-Mexico ITESM - Campus Monterrey},
    'my' => q{MYNIC Berhad},
    'mz' => q{Centro de Informatica de Universidade Eduardo Mondlane},
    'na' => q{Namibian Network Information Center},
    'nagoya' => q{GMO Registry, Inc.},
    'name' => q{VeriSign Information Services, Inc.},
    'navy' => q{United TLD Holdco Ltd.},
    'nc' => q{Office des Postes et Telecommunications},
    'ne' => q{SONITEL},
    'net' => q{VeriSign Global Registry Services},
    'network' => q{Trixy Manor, LLC},
    'neustar' => q{NeuStar, Inc.},
    'new' => q{Charleston Road Registry Inc.},
    'nexus' => q{Charleston Road Registry Inc.},
    'nf' => q{Norfolk Island Data Services},
    'ng' => q{Nigeria Internet Registration Association},
    'ngo' => q{Public Interest Registry},
    'nhk' => q{Japan Broadcasting Corporation (NHK)},
    'ni' => q{Universidad Nacional del Ingernieria Centro de Computo},
    'ninja' => q{United TLD Holdco Ltd.},
    'nl' => q{SIDN (Stichting Internet  Domeinregistratie Nederland)},
    'no' => q{UNINETT Norid A/S},
    'np' => q{Mercantile Communications Pvt. Ltd.},
    'nr' => q{CENPAC NET},
    'nra' => q{NRA Holdings Company, INC.},
    'nrw' => q{Minds + Machines GmbH},
    'ntt' => q{NIPPON TELEGRAPH AND TELEPHONE CORPORATION},
    'nu' => q{The IUSN Foundation},
    'nyc' => q{The City of New York by and through the New York City Department of Information Technology & Telecommunications},
    'nz' => q{InternetNZ},
    'okinawa' => q{BusinessRalliart inc.},
    'om' => q{Telecommunications Regulatory Authority (TRA)},
    'one' => q{One.com A/S},
    'ong' => q{Public Interest Registry},
    'onl' => q{I-REGISTRY Ltd., Niederlassung Deutschland},
    'ooo' => q{INFIBEAM INCORPORATION LIMITED},
    'org' => q{Public Interest Registry (PIR)},
    'organic' => q{Afilias Limited},
    'osaka' => q{Interlink Co., Ltd.},
    'otsuka' => q{Otsuka Holdings Co., Ltd.},
    'ovh' => q{OVH SAS},
    'pa' => q{Universidad Tecnologica de Panama},
    'paris' => q{City of Paris},
    'partners' => q{Magic Glen, LLC},
    'parts' => q{Sea Goodbye, LLC},
    'party' => q{Blue Sky Registry Limited},
    'pe' => q{Red Cientifica Peruana},
    'pf' => q{Gouvernement de la Polynsie franaise},
    'pg' => q{PNG DNS Administration Vice Chancellors Office The Papua New Guinea University of Technology},
    'ph' => q{PH Domain Foundation},
    'pharmacy' => q{National Association of Boards of Pharmacy},
    'photo' => q{Uniregistry, Corp.},
    'photography' => q{Sugar Glen, LLC},
    'photos' => q{Sea Corner, LLC},
    'physio' => q{PhysBiz Pty Ltd},
    'pics' => q{Uniregistry, Corp.},
    'pictures' => q{Foggy Sky, LLC},
    'pink' => q{Afilias Limited},
    'pizza' => q{Foggy Moon, LLC},
    'pk' => q{PKNIC},
    'pl' => q{Research and Academic Computer Network},
    'place' => q{Snow Galley, LLC},
    'plumbing' => q{Spring Tigers, LLC},
    'pm' => q{Association Franaise pour le Nommage Internet en Coopration (A.F.N.I.C.)},
    'pn' => q{Pitcairn Island Administration},
    'pohl' => q{Deutsche Vermgensberatung Aktiengesellschaft DVAG},
    'poker' => q{Afilias Domains No. 5 Limited},
    'porn' => q{ICM Registry PN LLC},
    'post' => q{Universal Postal Union},
    'pr' => q{Gauss Research Laboratory Inc.},
    'praxi' => q{Praxi S.p.A.},
    'press' => q{DotPress Inc.},
    'pro' => q{Registry Services Corporation dba RegistryPro},
    'prod' => q{Charleston Road Registry Inc.},
    'productions' => q{Magic Birch, LLC},
    'prof' => q{Charleston Road Registry Inc.},
    'properties' => q{Big Pass, LLC},
    'property' => q{Uniregistry, Corp.},
    'ps' => q{Ministry Of Telecommunications & Information Technology, Government Computer Center.},
    'pt' => q{Associao DNS.PT},
    'pub' => q{United TLD Holdco Ltd.},
    'pw' => q{Micronesia Investment and Development Corporation},
    'py' => q{NIC-PY},
    'qa' => q{Communications Regulatory Authority},
    'qpon' => q{dotCOOL, Inc.},
    'quebec' => q{PointQubec Inc},
    're' => q{Association Franaise pour le Nommage Internet en Coopration (A.F.N.I.C.)},
    'realtor' => q{Real Estate Domains LLC},
    'recipes' => q{Grand Island, LLC},
    'red' => q{Afilias Limited},
    'rehab' => q{United TLD Holdco Ltd.},
    'reise' => q{dotreise GmbH},
    'reisen' => q{New Cypress, LLC},
    'reit' => q{National Association of Real Estate Investment Trusts, Inc.},
    'ren' => q{Beijing Qianxiang Wangjing Technology Development Co., Ltd.},
    'rentals' => q{Big Hollow,LLC},
    'repair' => q{Lone Sunset, LLC},
    'report' => q{Binky Glen, LLC},
    'republican' => q{United TLD Holdco Ltd.},
    'rest' => q{Punto 2012 Sociedad Anonima Promotora de Inversion de Capital Variable},
    'restaurant' => q{Snow Avenue, LLC},
    'reviews' => q{United TLD Holdco, Ltd.},
    'rich' => q{I-REGISTRY Ltd., Niederlassung Deutschland},
    'rio' => q{Empresa Municipal de Informtica SA - IPLANRIO},
    'rip' => q{United TLD Holdco Ltd.},
    'ro' => q{National Institute for R&D in Informatics},
    'rocks' => q{United TLD Holdco, LTD.},
    'rodeo' => q{Top Level Domain Holdings Limited},
    'rs' => q{Serbian National Internet Domain Registry (RNIDS)},
    'rsvp' => q{Charleston Road Registry Inc.},
    'ru' => q{Coordination Center for TLD RU},
    'ruhr' => q{regiodot GmbH & Co. KG},
    'rw' => q{Rwanda Information Communication and Technology Association (RICTA)},
    'ryukyu' => q{BusinessRalliart inc.},
    'sa' => q{Communications and Information Technology Commission},
    'saarland' => q{dotSaarland GmbH},
    'sale' => q{United TLD Holdco, Ltd},
    'samsung' => q{SAMSUNG SDS CO., LTD},
    'sarl' => q{Delta Orchard, LLC},
    'sb' => q{Solomon Telekom Company Limited},
    'sc' => q{VCS Pty Ltd},
    'sca' => q{SVENSKA CELLULOSA AKTIEBOLAGET SCA (publ)},
    'scb' => q{The Siam Commercial Bank Public Company Limited ("SCB")},
    'schmidt' => q{SALM S.A.S.},
    'schule' => q{Outer Moon, LLC},
    'schwarz' => q{Schwarz Domains und Services GmbH & Co. KG},
    'science' => q{dot Science Limited},
    'scot' => q{Dot Scot Registry Limited},
    'sd' => q{Sudan Internet Society},
    'se' => q{The Internet Infrastructure Foundation},
    'services' => q{Fox Castle, LLC},
    'sew' => q{SEW-EURODRIVE GmbH & Co KG},
    'sexy' => q{Uniregistry, Corp.},
    'sg' => q{Singapore Network Information Centre (SGNIC) Pte Ltd},
    'sh' => q{Government of St. Helena},
    'shiksha' => q{Afilias Limited},
    'shoes' => q{Binky Galley, LLC},
    'shriram' => q{Shriram Capital Ltd.},
    'si' => q{Academic and Research Network of Slovenia (ARNES)},
    'singles' => q{Fern Madison, LLC},
    'sj' => q{UNINETT Norid A/S},
    'sk' => q{SK-NIC, a.s.},
    'sky' => q{Sky IP International Ltd, a company incorporated in England and Wales, operating via its registered Swiss branch},
    'sl' => q{Sierratel},
    'sm' => q{Telecom Italia San Marino S.p.A.},
    'sn' => q{Universite Cheikh Anta Diop NIC Senegal},
    'so' => q{Ministry of Post and Telecommunications},
    'social' => q{United TLD Holdco Ltd.},
    'software' => q{United TLD Holdco, Ltd},
    'sohu' => q{Sohu.com Limited},
    'solar' => q{Ruby Town, LLC},
    'solutions' => q{Silver Cover, LLC},
    'soy' => q{Charleston Road Registry Inc.},
    'space' => q{DotSpace Inc.},
    'spiegel' => q{SPIEGEL-Verlag Rudolf Augstein GmbH & Co. KG},
    'sr' => q{Telesur},
    'ss' => q{Not assigned},
    'st' => q{Tecnisys},
    'su' => q{Russian Institute for Development of Public Networks (ROSNIIROS)},
    'supplies' => q{Atomic Fields, LLC},
    'supply' => q{Half Falls, LLC},
    'support' => q{Grand Orchard, LLC},
    'surf' => q{Top Level Domain Holdings Limited},
    'surgery' => q{Tin Avenue, LLC},
    'suzuki' => q{SUZUKI MOTOR CORPORATION},
    'sv' => q{SVNet},
    'sx' => q{SX Registry SA B.V.},
    'sy' => q{National Agency for Network Services (NANS)},
    'sydney' => q{State of New South Wales, Department of Premier and Cabinet},
    'systems' => q{Dash Cypress, LLC},
    'sz' => q{University of Swaziland Department of Computer Science},
    'taipei' => q{Taipei City Government},
    'tatar' => q{Limited Liability Company "Coordination Center of Regional Domain of Tatarstan Republic"},
    'tattoo' => q{Uniregistry, Corp.},
    'tax' => q{Storm Orchard, LLC},
    'tc' => q{Melrex TC},
    'td' => q{Socit des tlcommunications du Tchad (SOTEL TCHAD)},
    'technology' => q{Auburn Falls, LLC},
    'tel' => q{Telnic Ltd.},
    'temasek' => q{Temasek Holdings (Private) Limited},
    'tf' => q{Association Franaise pour le Nommage Internet en Coopration (A.F.N.I.C.)},
    'tg' => q{Cafe Informatique et Telecommunications},
    'th' => q{Thai Network Information Center Foundation},
    'tienda' => q{Victor Manor, LLC},
    'tips' => q{Corn Willow, LLC},
    'tires' => q{Dog Edge, LLC},
    'tirol' => q{punkt Tirol GmbH},
    'tj' => q{Information Technology Center},
    'tk' => q{Telecommunication Tokelau Corporation (Teletok)},
    'tl' => q{Ministry of Transport and  Communications; National Division of  Information and Technology},
    'tm' => q{TM Domain Registry Ltd},
    'tn' => q{Agence Tunisienne d'Internet},
    'to' => q{Government of the Kingdom of Tonga H.R.H. Crown Prince Tupouto'a c/o Consulate of Tonga},
    'today' => q{Pearl Woods, LLC},
    'tokyo' => q{GMO Registry, Inc.},
    'tools' => q{Pioneer North, LLC},
    'top' => q{Jiangsu Bangning Science & Technology Co.,Ltd.},
    'town' => q{Koko Moon, LLC},
    'toys' => q{Pioneer Orchard, LLC},
    'tp' => q{-},
    'tr' => q{Middle East Technical University Department of Computer Engineering},
    'trade' => q{Elite Registry Limited},
    'training' => q{Wild Willow, LLC},
    'travel' => q{Tralliance Registry Management Company, LLC.},
    'trust' => q{Artemis Internet Inc},
    'tt' => q{University of the West Indies Faculty of Engineering},
    'tui' => q{TUI AG},
    'tv' => q{Ministry of Finance and Tourism},
    'tw' => q{Taiwan Network Information Center (TWNIC)},
    'tz' => q{Tanzania Network Information Centre (tzNIC)},
    'ua' => q{Hostmaster Ltd.},
    'ug' => q{Uganda Online Ltd.},
    'uk' => q{Nominet UK},
    'um' => q{Not assigned},
    'university' => q{Little Station, LLC},
    'uno' => q{Dot Latin LLC},
    'uol' => q{UBN INTERNET LTDA.},
    'us' => q{NeuStar, Inc.},
    'uy' => q{SeCIU - Universidad de la Republica},
    'uz' => q{Computerization and Information Technologies Developing Center UZINFOCOM},
    'va' => q{Holy See Secretariat of State Department of Telecommunications},
    'vacations' => q{Atomic Tigers, LLC},
    'vc' => q{Ministry of Telecommunications, Science, Technology and Industry},
    've' => q{Comisin Nacional de Telecomunicaciones (CONATEL)},
    'vegas' => q{Dot Vegas, Inc.},
    'ventures' => q{Binky Lake, LLC},
    'versicherung' => q{dotversicherung-registry GmbH},
    'vet' => q{United TLD Holdco, Ltd},
    'vg' => q{Telecommunications Regulatory Commission of the Virgin Islands},
    'vi' => q{Virgin Islands Public Telcommunications System c/o COBEX Internet Services},
    'viajes' => q{Black Madison, LLC},
    'video' => q{United TLD Holdco, Ltd},
    'villas' => q{New Sky, LLC},
    'vision' => q{Koko Station, LLC},
    'vlaanderen' => q{DNS.be vzw},
    'vn' => q{Ministry of Information and Communications of Socialist Republic of Viet Nam},
    'vodka' => q{Top Level Domain Holdings Limited},
    'vote' => q{Monolith Registry LLC},
    'voting' => q{Valuetainment Corp.},
    'voto' => q{Monolith Registry LLC},
    'voyage' => q{Ruby House, LLC},
    'vu' => q{Telecom Vanuatu Limited},
    'wales' => q{Nominet UK},
    'wang' => q{Zodiac Registry Limited},
    'watch' => q{Sand Shadow, LLC},
    'webcam' => q{dot Webcam Limited},
    'website' => q{DotWebsite Inc.},
    'wed' => q{Atgron, Inc.},
    'wedding' => q{Top Level Domain Holdings Limited},
    'wf' => q{Association Franaise pour le Nommage Internet en Coopration (A.F.N.I.C.)},
    'whoswho' => q{Who's Who Registry},
    'wien' => q{punkt.wien GmbH},
    'wiki' => q{Top Level Design, LLC},
    'williamhill' => q{William Hill Organization Limited},
    'wme' => q{William Morris Endeavor Entertainment, LLC},
    'work' => q{Top Level Domain Holdings Limited},
    'works' => q{Little Dynamite, LLC},
    'world' => q{Bitter Fields, LLC},
    'ws' => q{Government of Samoa Ministry of Foreign Affairs & Trade},
    'wtc' => q{World Trade Centers Association, Inc.},
    'wtf' => q{Hidden Way, LLC},
    'xxx' => q{ICM Registry LLC},
    'xyz' => q{XYZ.COM LLC},
    'yachts' => q{DERYachts, LLC},
    'yandex' => q{YANDEX, LLC},
    'ye' => q{TeleYemen},
    'yoga' => q{Top Level Domain Holdings Limited},
    'yokohama' => q{GMO Registry, Inc.},
    'youtube' => q{Charleston Road Registry Inc.},
    'yt' => q{Association Franaise pour le Nommage Internet en Coopration (A.F.N.I.C.)},
    'za' => q{ZA Domain Name Authority},
    'zip' => q{Charleston Road Registry Inc.},
    'zm' => q{Zambia Information and Communications Technology Authority (ZICTA)},
    'zone' => q{Outer Falls, LLC},
    'zuerich' => q{Kanton Zrich (Canton of Zurich)},
    'zw' => q{Postal and Telecommunications Regulatory Authority of Zimbabwe (POTRAZ)},
  }
);

my $flat_profile = flatten ( \%tld_profile );

sub flatten {
  my $hashref = shift;
  my %results;
  @results{ keys %{ $hashref->{$_} } } = values % { $hashref->{$_} }
    for ( keys %$hashref );
  return \%results;
}

sub check_type {
  my $type = shift;
  croak "unknown TLD type: $type" unless grep { $type eq $_ } TLD_TYPES;
  return 1;
}

=head1 PUBLIC METHODS

  Each public function/method is described here.
  These are how you should interact with this module.

=head3 C<< tlds >>

  This routine returns the tlds requested.

  my @all_tlds = tlds; #array of tlds
  my $all_tlds = tlds; #hashref of tlds and their descriptions

  my @cc_tlds = tlds('cc'); #array of just 'cc' type tlds
  my $cc_tlds = tlds('cc'); #hashref of just 'cc' type tlds and their descriptions

  Valid types are:
    cc                 - country code domains
    ccidn              - internationalized country code top-level domain 
    gtld_open          - generic domains that anyone can register
    gtld_restricted    - generic restricted registration domains
    gtld_new           - new gTLDs
    new_open           - recently added generic domains
    new_restricted     - new restricted registration domains
    reserved           - RFC2606 restricted names, not returned by tlds

=cut

sub tlds {
  my $type = shift;
  check_type ( $type ) if $type;
  my $results = $type ? 
    wantarray ? [ keys %{ $tld_profile{$type} } ] : 
      dclone ( $tld_profile{$type} ) :
	wantarray ? [ map { keys %$_ } values %tld_profile ] : 
	  $flat_profile;
  return wantarray ? @$results : $results;
}

=head3 C<< tld_exists >>

  This routine returns true if the given domain exists and false otherwise.

  die "no such domain" unless tld_exists($tld); #call without tld type 
  die "no such domain" unless tld_exists($tld, 'new_open'); #call with tld type

=cut

sub tld_exists {
  my ( $tld, $type )  = ( lc ( $_[0] ), $_[1] );
  check_type ( $type ) if $type;
  my $result = $type ? 
    $tld_profile{$type}{$tld} ? 1 : 0 :
    $flat_profile->{$tld} ? 1 : 0;
  return $result;
}

=head1 COPYRIGHT

  Copyright (c) 2003-2014 Alex Pavlovic, all rights reserved.  This program
  is free software; you can redistribute it and/or modify it under the same terms
  as Perl itself.

=head1 AUTHORS

  Alexander Pavlovic <alex.pavlovic@devradius.com>
  Ricardo SIGNES <rjbs@cpan.org>

=cut

1;