/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright 2009 by Sun Microsystems, Inc.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * $RCSfile: NelderMeadSolverImpl.java,v $
 * $Revision: 1.1 $
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

package com.sun.star.NLPSolver;

import com.sun.star.uno.XComponentContext;
import com.sun.star.lib.uno.helper.Factory;
import com.sun.star.lang.XSingleComponentFactory;
import com.sun.star.registry.XRegistryKey;
import com.sun.star.sheet.SolverConstraint;
import com.sun.star.sheet.SolverConstraintOperator;
import java.util.ArrayList;
import org.apache.commons.math.optimization.ConvergenceChecker;
import org.apache.commons.math.optimization.CostException;
import org.apache.commons.math.optimization.CostFunction;
import org.apache.commons.math.optimization.NelderMead;
import org.apache.commons.math.optimization.PointCostPair;


public final class NelderMeadSolverImpl extends BaseNLPSolver
   implements com.sun.star.lang.XServiceInfo
{
    private static final String m_implementationName = NelderMeadSolverImpl.class.getName();
    private static final String[] m_serviceNames = {
        "com.sun.star.sheet.Solver",
        "com.sun.star.beans.PropertySet"
    };

    public NelderMeadSolverImpl( XComponentContext context )
    {
        super(context, "Nelder-Mead");

        registerProperty(m_numberOfCorrections);
        registerProperty(m_epsilon);
        registerProperty(m_gradientStepSize);
    }

    public static XSingleComponentFactory __getComponentFactory( String sImplementationName ) {
        XSingleComponentFactory xFactory = null;

        if ( sImplementationName.equals( m_implementationName ) )
            xFactory = Factory.createComponentFactory(NelderMeadSolverImpl.class, m_serviceNames);
        return xFactory;
    }

    public static boolean __writeRegistryServiceInfo( XRegistryKey xRegistryKey ) {
        return Factory.writeRegistryServiceInfo(m_implementationName,
                                                m_serviceNames,
                                                xRegistryKey);
    }

    // com.sun.star.lang.XServiceInfo:
    public String getImplementationName() {
         return m_implementationName;
    }

    public boolean supportsService( String sService ) {
        int len = m_serviceNames.length;

        for( int i=0; i < len; i++) {
            if (sService.equals(m_serviceNames[i]))
                return true;
        }
        return false;
    }

    public String[] getSupportedServiceNames() {
        return m_serviceNames;
    }

    // com.sun.star.sheet.XSolver:
    private int run;

    private PropertyInfo<Integer> m_numberOfCorrections = new PropertyInfo<Integer>("NumberOfCorrections", 7, "Number of Correction (3-7)"); //parameter "m"
    private PropertyInfo<Double> m_epsilon = new PropertyInfo<Double>("Epsilon", 0.000001, "Epsilon"); //precision
    private PropertyInfo<Double> m_gradientStepSize = new PropertyInfo<Double>("GradientStepSize", 0.0001, "Gradient Stepsize"); //parameter "h"
    
    private ArrayList<ExtSolverConstraint> m_extConstraints = new ArrayList<ExtSolverConstraint>();

    private class SolverCostFunction implements CostFunction {
        public double cost(double[] x) throws CostException {
            for (int i = 0; i < m_variableCount; ++i)
                m_variableCells[i].setValue(x[i]);

            if (m_objectiveCell.getError() != 0)
                throw new CostException(new RuntimeException());

            return m_objectiveCell.getValue() * ( m_maximize ? -1.0 : 1.0 ) + (10 * run++) * getPenalty();
        }
    }

    private static class ValueChecker implements ConvergenceChecker {
        public ValueChecker(double threshold) {
            this.threshold = threshold;
        }

        public boolean converged(PointCostPair[] simplex) {
            PointCostPair smallest = simplex[0];
            PointCostPair largest  = simplex[simplex.length - 1];
            return (largest.getCost() - smallest.getCost()) < threshold;
        }

        private double threshold;
    }

    @Override
    public void setConstraints(SolverConstraint[] constraints) {
        super.setConstraints(constraints);
        
        m_extConstraints.clear();
        for (int i = 0; i < m_constraintCount; i++)
            m_extConstraints.add(new ExtSolverConstraint(m_constraintCells[i], m_constraints[i].Operator, m_constraints[i].Right));
    }

    public double getPenalty() {
        double result = 0.0;

        for (int i = 0; i < m_constraintCount; i++) {
            ExtSolverConstraint constraint = m_extConstraints.get(i);
            switch (constraint.Operator.getValue()) {
                case SolverConstraintOperator.EQUAL_value:
                    result += Math.pow(Math.abs(constraint.Data - constraint.getLeftValue()), 2);
                    break;
                case SolverConstraintOperator.GREATER_EQUAL_value:
                    result += Math.pow(Math.min(0.0, constraint.Data - constraint.getLeftValue()), 2);
                    break;
                case SolverConstraintOperator.LESS_EQUAL_value:
                    result += Math.pow(Math.min(0.0, constraint.getLeftValue() - constraint.Data), 2);
                    break;
            }
        }

        return result;
    }

    public void solve() {
        initializeSolve();

        double[] low = new double[m_variableCount];
        double[] high = new double[m_variableCount];
        for (int i = 0; i < m_variableCount; i++) {
            low[i] = 0.1;
            high[i] = 1.0;
        }

        run = 1;

        CostFunction func = new SolverCostFunction();
        NelderMead nm = new NelderMead();
        try {
            PointCostPair optimum = nm.minimize( func, 1000, new ValueChecker(1.0e-6), low, high );
            m_success = true;
            m_functionValue = optimum.getCost() * ( m_maximize ? -1.0 : 1.0 );
            m_currentParameters = optimum.getPoint();
        } catch (Exception e) {
            // m_success remains false
        }

        finalizeSolve();
    }

}
