# - Try to find TAO
# Once done this will define
#
#  TAO_FOUND        - system has TAO
#  TAO_INCLUDE_DIR  - include directories for TAO
#  TAO_LIBARIES     - libraries for TAO
#  TAO_DIR          - directory where TAO is built
#  TAO_VERSION      - version of TAO
#
# Assumes that PETSC_DIR and PETSC_ARCH has been set by
# alredy calling find_package(PETSc)

#=============================================================================
# Copyright (C) 2010-2012 Garth N. Wells, Anders Logg and Johannes Ring
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in
#    the documentation and/or other materials provided with the
#    distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
# COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
# ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#=============================================================================

message(STATUS "Checking for package 'TAO'")

# Set debian_arches (PETSC_ARCH for Debian-style installations)
foreach (debian_arches linux kfreebsd)
  if ("${CMAKE_BUILD_TYPE}" STREQUAL "Debug")
    set(DEBIAN_FLAVORS ${debian_arches}-gnu-c-debug ${debian_arches}-gnu-c-opt ${DEBIAN_FLAVORS})
  else()
    set(DEBIAN_FLAVORS ${debian_arches}-gnu-c-opt ${debian_arches}-gnu-c-debug ${DEBIAN_FLAVORS})
  endif()
endforeach()

# List of possible locations for TAO_DIR
set(tao_dir_locations "")
list(APPEND tao_dir_locations "/usr/local/lib/tao")
list(APPEND tao_dir_locations "$ENV{HOME}/tao")

# Add other possible locations for TAO_DIR
set(_SYSTEM_LIB_PATHS "${CMAKE_SYSTEM_LIBRARY_PATH};${CMAKE_PLATFORM_IMPLICIT_LINK_DIRECTORIES}")
string(REGEX REPLACE ":" ";" libdirs ${_SYSTEM_LIB_PATHS})
foreach (libdir ${libdirs})
  get_filename_component(tao_dir_location "${libdir}/" PATH)
  list(APPEND tao_dir_locations ${tao_dir_location})
endforeach()

# Try to figure out TAO_DIR by finding tao.h
find_path(TAO_DIR include/tao.h
  HINTS ${TAO_DIR} $ENV{TAO_DIR}
  PATHS ${tao_dir_locations}
  DOC "TAO directory")

# Report result of search for TAO_DIR
if (DEFINED TAO_DIR)
  message(STATUS "TAO_DIR is ${TAO_DIR}")
else()
  message(STATUS "TAO_DIR is empty")
endif()

# Get variables from TAO configuration
if (TAO_DIR)

  find_library(TAO_LIBRARY
    NAMES tao
    HINTS ${TAO_DIR}/lib $ENV{TAO_DIR}/lib  ${TAO_DIR}/${PETSC_ARCH}/lib $ENV{TAO_DIR}/$ENV{PETSC_ARCH}/lib
    NO_DEFAULT_PATH
    DOC "The TAO library"
    )
  find_library(TAO_LIBRARY
    NAMES tao
    DOC "The TAO library"
    )
  mark_as_advanced(TAO_LIBRARY)

  # Create a temporary Makefile to probe the TAO configuration
  set(tao_config_makefile ${PROJECT_BINARY_DIR}/Makefile.tao)
  file(WRITE ${tao_config_makefile}
"# This file was autogenerated by FindTAO.cmake
TAO_DIR  = ${TAO_DIR}
PETSC_ARCH = ${PETSC_ARCH}
PETSC_DIR = ${PETSC_DIR}
include ${TAO_DIR}/conf/tao_base
showvar :
	-@echo -n \${\${VARIABLE}}
")

  # Define macro for getting TAO variables from Makefile
  macro(TAO_GET_VARIABLE var name)
    set(${var} "NOTFOUND" CACHE INTERNAL "Cleared" FORCE)
    execute_process(COMMAND ${CMAKE_MAKE_PROGRAM} --no-print-directory -f ${tao_config_makefile} showvar VARIABLE=${name}
      OUTPUT_VARIABLE ${var}
      RESULT_VARIABLE tao_return)
  endmacro()

  # Call macro to get the TAO variables
  tao_get_variable(TAO_INCLUDE TAO_INCLUDE)
  tao_get_variable(TAO_EXTERNAL_LIB TAO_EXTERNAL_LIB)

  # Remove temporary Makefile
  file(REMOVE ${tao_config_makefile})

  # Extract include paths and libraries from compile command line
  include(ResolveCompilerPaths)
  resolve_includes(TAO_INCLUDE_DIRS "${TAO_INCLUDE}")
  resolve_libraries(TAO_EXTERNAL_LIBRARIES "${TAO_EXTERNAL_LIB}")

  # Add variables to CMake cache and mark as advanced
  set(TAO_INCLUDE_DIRS ${TAO_INCLUDE_DIRS} CACHE STRING "TAO include paths." FORCE)
  set(TAO_LIBRARIES ${TAO_LIBRARY} CACHE STRING "TAO libraries." FORCE)
  mark_as_advanced(TAO_INCLUDE_DIRS TAO_LIBRARIES)
endif()

if (DOLFIN_SKIP_BUILD_TESTS)
  set(TAO_TEST_RUNS TRUE)
  set(TAO_VERSION "UNKNOWN")
  set(TAO_VERSION_OK TRUE)
elseif (TAO_LIBRARIES AND TAO_INCLUDE_DIRS)

  # Set flags for building test program
  set(CMAKE_REQUIRED_INCLUDES ${TAO_INCLUDE_DIRS} ${PETSC_INCLUDE_DIRS})
  set(CMAKE_REQUIRED_LIBRARIES ${TAO_LIBRARIES} ${PETSC_LIBRARIES})

  # Add MPI variables if MPI has been found
  if (MPI_C_FOUND)
    set(CMAKE_REQUIRED_INCLUDES  ${CMAKE_REQUIRED_INCLUDES} ${MPI_C_INCLUDE_PATH} ${BLAS_INCLUDE_PATH})
    set(CMAKE_REQUIRED_LIBRARIES ${CMAKE_REQUIRED_LIBRARIES} ${MPI_C_LIBRARIES} ${BLAS_LIBRARIES})
    set(CMAKE_REQUIRED_FLAGS     "${CMAKE_REQUIRED_FLAGS}${MPI_C_LIBRARIES} ${BLAS_LIBRARIES}")
  endif()

  # Run TAO test program
  set(TAO_TEST_LIB_CPP
    "${CMAKE_CURRENT_BINARY_DIR}/CMakeFiles/tao_test_lib.cpp")
  file(WRITE ${TAO_TEST_LIB_CPP} "
#include \"petsc.h\"
#include \"tao.h\"
int main()
{
  PetscErrorCode ierr;
  int argc = 0;
  char** argv = NULL;
  ierr = TaoInitialize(&argc, &argv, PETSC_NULL, PETSC_NULL);
  TaoSolver tao;
  ierr = TaoCreate(PETSC_COMM_SELF, &tao); CHKERRQ(ierr);
  //ierr = TaoSetFromOptions(tao); CHKERRQ(ierr);
  ierr = TaoDestroy(&tao); CHKERRQ(ierr);
  ierr = TaoFinalize(); CHKERRQ(ierr);
  return 0;
}
")

  try_run(
    TAO_TEST_LIB_EXITCODE
    TAO_TEST_LIB_COMPILED
    ${CMAKE_CURRENT_BINARY_DIR}
    ${TAO_TEST_LIB_CPP}
    CMAKE_FLAGS
      "-DINCLUDE_DIRECTORIES:STRING=${CMAKE_REQUIRED_INCLUDES}"
      "-DLINK_LIBRARIES:STRING=${CMAKE_REQUIRED_LIBRARIES}"
    COMPILE_OUTPUT_VARIABLE TAO_TEST_LIB_COMPILE_OUTPUT
    RUN_OUTPUT_VARIABLE TAO_TEST_LIB_OUTPUT
    )

  if (TAO_TEST_LIB_COMPILED AND TAO_TEST_LIB_EXITCODE EQUAL 0)
    message(STATUS "Performing test TAO_TEST_RUNS - Success")
    set(TAO_TEST_RUNS TRUE)
  else()
    message(STATUS "Performing test TAO_TEST_RUNS - Failed")

    # Test program does not run - try adding TAO 3rd party libs and test again
    list(APPEND CMAKE_REQUIRED_LIBRARIES ${TAO_EXTERNAL_LIBRARIES})

    try_run(
      TAO_TEST_3RD_PARTY_LIBS_EXITCODE
      TAO_TEST_3RD_PARTY_LIBS_COMPILED
      ${CMAKE_CURRENT_BINARY_DIR}
      ${TAO_TEST_LIB_CPP}
      CMAKE_FLAGS
        "-DINCLUDE_DIRECTORIES:STRING=${CMAKE_REQUIRED_INCLUDES}"
	"-DLINK_LIBRARIES:STRING=${CMAKE_REQUIRED_LIBRARIES}"
      COMPILE_OUTPUT_VARIABLE TAO_TEST_3RD_PARTY_LIBS_COMPILE_OUTPUT
      RUN_OUTPUT_VARIABLE TAO_TEST_3RD_PARTY_LIBS_OUTPUT
      )

    if (TAO_TEST_3RD_PARTY_LIBS_COMPILED AND TAO_TEST_3RD_PARTY_LIBS_EXITCODE EQUAL 0)
      message(STATUS "Performing test TAO_TEST_3RD_PARTY_LIBS_RUNS - Success")
      set(TAO_LIBRARIES ${TAO_LIBRARIES} ${TAO_EXTERNAL_LIBRARIES}
	CACHE STRING "TAO libraries." FORCE)
      set(TAO_TEST_RUNS TRUE)
    else()
      message(STATUS "Performing test TAO_TEST_3RD_PARTY_LIBS_RUNS - Failed")
    endif()
  endif()
endif()

# Standard package handling
include(FindPackageHandleStandardArgs)
find_package_handle_standard_args(TAO
  "TAO could not be found. Be sure to set TAO_DIR, PETSC_DIR, and PETSC_ARCH."
  TAO_LIBRARIES TAO_DIR TAO_INCLUDE_DIRS TAO_TEST_RUNS)
