/** @file
 * @brief HID report descriptor - usage types
 *
 * Copyright (C) 2010 Nikolai Kondrashov
 *
 * This file is part of hidrd.
 *
 * Hidrd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Hidrd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with hidrd; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @author Nikolai Kondrashov <spbnick@gmail.com>
 *
 * @(#) $Id: type.h 413 2010-05-12 19:31:17Z spb_nick $
 */

#ifndef __HIDRD_USAGE_TYPE_H__
#define __HIDRD_USAGE_TYPE_H__

#include <stdbool.h>
#include <stdint.h>
#include "hidrd/cfg.h"
#include "hidrd/util/tkn.h"

#ifdef __cplusplus
extern "C" {
#endif

/** Usage type bit indexes */
typedef enum hidrd_usage_type_idx {
    /* Controls */
    HIDRD_USAGE_TYPE_IDX_LC,    /**< Linear control */
    HIDRD_USAGE_TYPE_IDX_OOC,   /**< On/off control */
    HIDRD_USAGE_TYPE_IDX_MC,    /**< Momentary control */
    HIDRD_USAGE_TYPE_IDX_OSC,   /**< One-shot control */
    HIDRD_USAGE_TYPE_IDX_RTC,   /**< Re-trigger control */
    /* Data */
    HIDRD_USAGE_TYPE_IDX_SEL,   /**< Selector */
    HIDRD_USAGE_TYPE_IDX_SV,    /**< Static value */
    HIDRD_USAGE_TYPE_IDX_SF,    /**< Static flag */
    HIDRD_USAGE_TYPE_IDX_DV,    /**< Dynamic value */
    HIDRD_USAGE_TYPE_IDX_DF,    /**< Dynamic flag */
    /* Collections */
    HIDRD_USAGE_TYPE_IDX_NARY,  /**< Named array */
    HIDRD_USAGE_TYPE_IDX_CA,    /**< Application collection */
    HIDRD_USAGE_TYPE_IDX_CL,    /**< Logical collection */
    HIDRD_USAGE_TYPE_IDX_CP,    /**< Physical collection */
    HIDRD_USAGE_TYPE_IDX_US,    /**< Usage switch */
    HIDRD_USAGE_TYPE_IDX_UM,    /**< Usage modifier */
} hidrd_usage_type_idx;

#define HIDRD_USAGE_TYPE_IDX_MIN    HIDRD_USAGE_TYPE_IDX_LC
#define HIDRD_USAGE_TYPE_IDX_MAX    HIDRD_USAGE_TYPE_IDX_UM
#define HIDRD_USAGE_TYPE_IDX_NUM    (HIDRD_USAGE_TYPE_IDX_MAX - \
                                     HIDRD_USAGE_TYPE_IDX_MIN + 1)

static inline bool
hidrd_usage_type_idx_valid(hidrd_usage_type_idx idx)
{
    return (idx <= HIDRD_USAGE_TYPE_IDX_MAX);
}

#ifdef HIDRD_WITH_TOKENS
/* Declare type bit index <-> token conversion functions */
HIDRD_TKN_CONV_DECLS(usage_type_idx);
#endif /* HIDRD_WITH_TOKENS */

#ifdef HIDRD_WITH_NAMES
/**
 * Retrieve a usage type bit index name.
 *
 * @param idx   Usage type bit index.
 *
 * @return Constant usage type bit index string.
 */
extern const char *
hidrd_usage_type_idx_name(hidrd_usage_type_idx idx);
#endif

/** Usage type bits */
typedef enum hidrd_usage_type {
#define TYPE(_NAME) \
    HIDRD_USAGE_TYPE_##_NAME    = 1 << HIDRD_USAGE_TYPE_IDX_##_NAME
    /* Controls */
    TYPE(LC),       /**< Linear control */
    TYPE(OOC),      /**< On/off control */
    TYPE(MC),       /**< Momentary control */
    TYPE(OSC),      /**< One-shot control */
    TYPE(RTC),      /**< Re-trigger control */
    /* Data */
    TYPE(SEL),      /**< Selector */
    TYPE(SV),       /**< Static value */
    TYPE(SF),       /**< Static flag */
    TYPE(DV),       /**< Dynamic value */
    TYPE(DF),       /**< Dynamic flag */
    /* Collections */
    TYPE(NARY),     /**< Named array */
    TYPE(CA),       /**< Application collection */
    TYPE(CL),       /**< Logical collection */
    TYPE(CP),       /**< Physical collection */
    TYPE(US),       /**< Usage switch */
    TYPE(UM),       /**< Usage modifier */
#undef TYPE
} hidrd_usage_type;

/**
 * Check if a usage type is valid.
 *
 * @param type  Type to check.
 *
 * @return True if the type is valid, false otherwise.
 */
extern bool hidrd_usage_type_valid(hidrd_usage_type type);

/** Usage type set (bitmask) */
typedef uint32_t    hidrd_usage_type_set;

/** Empty type set */
#define HIDRD_USAGE_TYPE_SET_EMPTY  0

/** Bitmask with non-valid type set bits set */
#define HIDRD_USAGE_TYPE_SET_NOT_MASK \
    ((~(hidrd_usage_type)0) << (HIDRD_USAGE_TYPE_IDX_MAX + 1))
/** Bitmask with valid type set bits set */
#define HIDRD_USAGE_TYPE_SET_MASK \
    (~HIDRD_USAGE_TYPE_SET_NOT_MASK)

/**
 * Check if a usage type set is valid.
 *
 * @param set   Type set to check.
 *
 * @return True is the set is valid, false otheriwse.
 */
static inline bool
hidrd_usage_type_set_valid(hidrd_usage_type_set set)
{
    return (set & HIDRD_USAGE_TYPE_SET_NOT_MASK) == 0;
}

#ifdef HIDRD_WITH_NAMES
/**
 * Format a string description of a usage type set (bitmask).
 *
 * @param set   Usage type set.
 *
 * @return Dynamically allocated description string, or NULL if failed to
 *         allocate memory; will be an empty string if the set is empty.
 */
extern char *hidrd_usage_type_set_desc_str(hidrd_usage_type_set set);
#endif /* HIDRD_WITH_NAMES */

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* __HIDRD_USAGE_TYPE_H__ */
