package org.apache.commons.openpgp;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * Interface for signing data with OpenPGP.
 *
 * @author <a href="mailto:brett@apache.org">Brett Porter</a>
 * @todo perhaps should have different interface methods for the default key
 * @todo should the exception be a signature exception instead of a common one?
 */
public interface OpenPgpSigner
{
    String ROLE = OpenPgpSigner.class.getName();

    /**
     * Sign a piece of data with the given key.
     *
     * @param data         the data to sign
     * @param signedOutput the signed output data
     * @param keyId        the key ID of the key used to sign it
     * @param keyRing      the keyring containing the key above
     * @param asciiArmor   whether to ascii armor the output
     */
    void sign( InputStream data, OutputStream signedOutput, String keyId, KeyRing keyRing, boolean asciiArmor )
        throws OpenPgpException;

    /**
     * Sign a piece of data with the given key, storing the signature in a detached output.
     *
     * @param data       the data to sign
     * @param signature  the detached signature
     * @param keyId      the key ID of the key used to sign it
     * @param keyRing    the keyring containing the key above
     * @param asciiArmor whether to ascii armor the output
     */
    void detachedSign( InputStream data, OutputStream signature, String keyId, KeyRing keyRing, boolean asciiArmor )
        throws OpenPgpException, IOException;
}
