/*!
 * \file
 * \brief Pulse shaping classes test program
 * \author Adam Piatyszek
 *
 * -------------------------------------------------------------------------
 *
 * Copyright (C) 1995-2013  (see AUTHORS file for a list of contributors)
 *
 * This file is part of IT++ - a C++ library of mathematical, signal
 * processing, speech processing, and communications classes and functions.
 *
 * IT++ is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * IT++ is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with IT++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * -------------------------------------------------------------------------
 */

#include <itpp/itcomm.h>
#include "gtest/gtest.h"

using namespace std;
using namespace itpp;

static
const double tol = 1e-4;
static
void assert_vec_p(const vec &ref, const vec &act, int line)
{
  ASSERT_EQ(ref.length(), act.length()) << line;
  for (int n = 0; n < ref.length(); ++n) {
    ASSERT_NEAR(ref(n), act(n), tol) << line;
  }
}
#define assert_vec(ref, act) assert_vec_p(ref, act, __LINE__)
static
void assert_cvec_p(const cvec &ref, const cvec &act, int line)
{
  ASSERT_EQ(ref.length(), act.length()) << line;
  for (int n = 0; n < ref.length(); ++n) {
    ASSERT_NEAR(ref(n).real(), act(n).real(), tol) << line;
    ASSERT_NEAR(ref(n).imag(), act(n).imag(), tol) << line;
  }
}
#define assert_cvec(ref, act) assert_cvec_p(ref, act, __LINE__)


TEST(PulseShape, All)
{
  // Test of pulse shaping routines
  RNG_reset(0);

  BPSK bpsk;
  vec symbols, samples, rsymbols;
  Root_Raised_Cosine<double> rrc_tx(0.5), rrc_rx(0.5);
  Raised_Cosine<double> rc_tx(0.5);

  bpsk.modulate_bits(randb(20), symbols);
  samples = rrc_tx.shape_symbols(symbols);
  rsymbols = rrc_rx.shape_samples(samples);

  // Root Raised Cosine; real input
  vec ref = "0.00107181 -0.0027027 -0.00581667 -0.00696875 -0.00530516 -0.000815537 0.00546883 0.0115738 0.0150053 "
  "0.0134306 0.00546883 -0.00861562 -0.0265258 -0.0440088 -0.0554523 -0.0549318 -0.0375132 -0.000532387 0.0554523 "
  "0.126288 0.204577 0.280729 0.344536 0.386975 0.401856 0.386975 0.344536 0.280729 0.204577 0.126288 0.0554523 "
  "-0.000532387 -0.0375132 -0.0549318 -0.0554523 -0.0440088 -0.0265258 -0.00861562 0.00546883 0.0134306 0.0150053 "
  "0.0115738 0.00546883 -0.000815537 -0.00530516 -0.00696875 -0.00581667 -0.0027027 0.00107181";
  assert_vec(ref, rrc_tx.get_pulse_shape());
  ref = "-0.00107181 0.0027027 0.00581667 0.00696875 0.00530516 0.000815537 -0.00546883 -0.0115738 -0.0160771 "
  "-0.0107279 0.000347842 0.0155844 0.031831 0.0448243 0.0499834 0.0433579 0.0214361 -0.0101955 -0.0551044 "
  "-0.110704 -0.172746 -0.235905 -0.294553 -0.343617 -0.38042 -0.397171 -0.399641 -0.391433 -0.377324 -0.362193 "
  "-0.350005 -0.343085 -0.340763 -0.347645 -0.355822 -0.361361 -0.361408 -0.355208 -0.344536 -0.333368 -0.327901 "
  "-0.326952 -0.338719 -0.36384 -0.398545 -0.434626 -0.460562 -0.463676 -0.43401 -0.354878 -0.238753 -0.094032 "
  "0.063662 0.214849 0.339415 0.420138 0.446872 0.414732 0.327782 0.200912 0.0530516 -0.0956631 -0.227815 -0.33173 "
  "-0.401856 -0.442221 -0.461257 -0.465795 -0.462207 -0.453488 -0.438686 -0.413668 -0.373989 -0.304869 -0.216877 "
  "-0.112894 1.38778e-17 0.112894 0.216877 0.304869 0.373989 0.413668 0.438686 0.453488 0.462207 0.465795 0.461257 "
  "0.442221 0.401856 0.33173 0.227815 0.0956631 -0.0530516 -0.200912 -0.327782 -0.414732 -0.446872 -0.420138 "
  "-0.339415 -0.214849 -0.063662 0.094032 0.238753 0.354878 0.431866 0.469082 0.472195 0.448563 0.409155 0.365471 "
  "0.327782 0.303804 0.296819 0.30921 0.339415 0.379408 0.419765 0.450195 0.461257 0.445934 0.399712 0.333422 "
  "0.239448 0.125201 -1.38778e-17 -0.125201 -0.239448 -0.333422 -0.399712 -0.445934 -0.461257 -0.450195 -0.419765 "
  "-0.379408 -0.339415 -0.30921 -0.296819 -0.303804 -0.327782 -0.365471 -0.409155 -0.448563 -0.472195 -0.469082 "
  "-0.431866 -0.354878 -0.238753 -0.094032 0.063662 0.214849 0.339415 0.420138 0.446872 0.414732 0.327782 0.200912 "
  "0.0530516 -0.0956631 -0.227815 -0.33173";
  assert_vec(ref, samples);
  ref = "-1.14877e-06 5.79353e-06 5.16411e-06 -1.65031e-05 -6.01302e-05 -0.000107998 -0.000126412 -7.86767e-05 "
  "5.33562e-05 0.000260324 0.000459271 0.000528627 0.00034915 -0.000140663 -0.000892236 -0.00171907 -0.00231299 "
  "-0.00229884 -0.00137958 0.000562355 0.00336609 0.00655046 0.00935825 0.0108978 0.0103664 0.00732923 0.00192998 "
  "-0.00492754 -0.0115725 -0.0157611 -0.0150763 -0.00748811 0.00802133 0.0308814 0.0586051 0.0867172 0.109087 "
  "0.118648 0.108416 0.0726618 0.00800958 -0.0857266 -0.205223 -0.343836 -0.492374 -0.640337 -0.777394 -0.894844 "
  "-0.986783 -1.05073 -1.08773 -1.10165 -1.09811 -1.08321 -1.06236 -1.03947 -1.01677 -0.995122 -0.974794 -0.956374 "
  "-0.941541 -0.933381 -0.936049 -0.953742 -0.989175 -1.04189 -1.10689 -1.17398 -1.22825 -1.25181 -1.22656 -1.13759 "
  "-0.976659 -0.744793 -0.453718 -0.125306 0.210732 0.520675 0.771508 0.935703 0.995194 0.943845 0.788112 0.545574 "
  "0.241958 -0.0928153 -0.428764 -0.739431 -1.00437 -1.21023 -1.35063 -1.42483 -1.43609 -1.38985 -1.29244 -1.15025 "
  "-0.96958 -0.756879 -0.519122 -0.264014 -2.08438e-17 0.264014 0.519122 0.756879 0.969578 1.15026 1.29245 1.38982 "
  "1.43597 1.42462 1.35038 1.21007 1.00447 0.739946 0.429677 0.0938891 -0.241199 -0.545747 -0.78977 -0.947205 -0.999873 "
  "-0.940561 -0.774727 -0.520079 -0.204349 0.138547 0.473327 0.768307 0.999707 1.15454 1.23178 1.24143 1.20186 "
  "1.13612 1.06763 1.01617 0.994738 1.00769 1.05044 1.11059 1.17023 1.20906 1.20773 1.1509 1.02972 0.843259 0.598789 "
  "0.310972 2.53974e-17 -0.310972 -0.598789 -0.843259 -1.02972 -1.1509 -1.20773 -1.20906 -1.17023 -1.11059 -1.05044 -1.00769";
  assert_vec(ref, rsymbols);

  samples = rc_tx.shape_symbols(symbols);

  // Raised Cosine; real input
  ref = "8.95103e-34 0.00113767 0.00477279 0.0105661 0.0171489 0.0221856 0.0227496 0.015992 -1.29939e-17 -0.0253289 "
  "-0.0576127 -0.0917169 -0.120042 -0.133416 -0.122502 -0.0795251 3.06162e-17 0.115879 0.262504 0.428983 0.600211 "
  "0.758749 0.887236 0.970942 1 0.970942 0.887236 0.758749 0.600211 0.428983 0.262504 0.115879 3.06162e-17 "
  "-0.0795251 -0.122502 -0.133416 -0.120042 -0.0917169 -0.0576127 -0.0253289 -1.29939e-17 0.015992 0.0227496 "
  "0.0221856 0.0171489 0.0105661 0.00477279 0.00113767 8.95103e-34";
  assert_vec(ref, rc_tx.get_pulse_shape());
  ref = "-8.95103e-34 -0.00113767 -0.00477279 -0.0105661 -0.0171489 -0.0221856 -0.0227496 -0.015992 1.29939e-17 "
  "0.0241912 0.0528399 0.0811508 0.102893 0.11123 0.0997521 0.0635331 -1.76223e-17 -0.0916881 -0.209664 "
  "-0.347832 -0.497318 -0.647518 -0.787484 -0.907409 -1 -1.06263 -1.0969 -1.10658 -1.09753 -1.0765 -1.04999 "
  "-1.02329 -1 -0.980829 -0.964853 -0.952033 -0.943189 -0.940413 -0.946876 -0.965975 -1 -1.04975 -1.11237 "
  "-1.17878 -1.23472 -1.26218 -1.24215 -1.15815 -1 -0.767338 -0.47214 -0.137385 0.205787 0.522147 0.777324 "
  "0.942787 1 0.945062 0.78687 0.543279 0.240084 -0.0930137 -0.426641 -0.735354 -1 -1.20653 -1.34783 -1.42448 "
  "-1.44051 -1.40124 -1.31188 -1.17682 -1 -0.786012 -0.541866 -0.276447 3.46945e-17 0.276447 0.541866 0.786012 "
  "1 1.17682 1.31188 1.40124 1.44051 1.42448 1.34783 1.20653 1 0.735354 0.426641 0.0930137 -0.240084 -0.543279 "
  "-0.78687 -0.945062 -1 -0.942787 -0.777324 -0.522147 -0.205787 0.137385 0.47214 0.767338 1 1.15587 1.23261 "
  "1.24105 1.20042 1.13441 1.06687 1.01777 1 1.0155 1.05733 1.11328 1.16612 1.19668 1.18711 1.12389 1 0.81572 "
  "0.57782 0.299687 6.93889e-18 -0.299687 -0.57782 -0.81572 -1 -1.12389 -1.18711 -1.19668 -1.16612 -1.11328 "
  "-1.05733 -1.0155 -1 -1.01777 -1.06687 -1.13441 -1.20042 -1.24105 -1.23261 -1.15587 -1 -0.767338 -0.47214 "
  "-0.137385 0.205787 0.522147 0.777324 0.942787 1 0.945062 0.78687 0.543279 0.240084 -0.0930137 -0.426641 -0.735354";
  assert_vec(ref, samples);

  QPSK qpsk;
  cvec csymbols, csamples, crsymbols;
  Root_Raised_Cosine<complex<double> > crrc_tx(0.5), crrc_rx(0.5);
  Raised_Cosine<complex<double> > crc_tx(0.5);

  qpsk.modulate_bits(randb(40), csymbols);
  csamples = crrc_tx.shape_symbols(csymbols);
  crsymbols = crrc_rx.shape_samples(csamples);

  // Root Raised Cosine; complex input
  ref = "0.00107181 -0.0027027 -0.00581667 -0.00696875 -0.00530516 -0.000815537 0.00546883 0.0115738 0.0150053 "
  "0.0134306 0.00546883 -0.00861562 -0.0265258 -0.0440088 -0.0554523 -0.0549318 -0.0375132 -0.000532387 0.0554523 "
  "0.126288 0.204577 0.280729 0.344536 0.386975 0.401856 0.386975 0.344536 0.280729 0.204577 0.126288 0.0554523 "
  "-0.000532387 -0.0375132 -0.0549318 -0.0554523 -0.0440088 -0.0265258 -0.00861562 0.00546883 0.0134306 0.0150053 "
  "0.0115738 0.00546883 -0.000815537 -0.00530516 -0.00696875 -0.00581667 -0.0027027 0.00107181";
  assert_vec(ref, crrc_tx.get_pulse_shape());
  cvec ref_c = "0.00107181+0i -0.0027027+0i -0.00581667+0i -0.00696875+0i -0.00530516+0i -0.000815537+0i 0.00546883+0i "
  "0.0115738+0i 0.0150053-0.00107181i 0.0134306+0.0027027i 0.00546883+0.00581667i -0.00861562+0.00696875i "
  "-0.0265258+0.00530516i -0.0440088+0.000815537i -0.0554523-0.00546883i -0.0549318-0.0115738i -0.0364414-0.0150053i "
  "-0.00323508-0.0134306i 0.0496356-0.00546883i 0.119319+0.00861562i 0.199272+0.0265258i 0.279913+0.0440088i "
  "0.350005+0.0554523i 0.398549+0.0549318i 0.415789+0.0375132i 0.403109+0.000532387i 0.355822-0.0554523i "
  "0.279082-0.126288i 0.183357-0.204577i 0.083095-0.280729i -0.00546883-0.344536i -0.067038-0.386975i "
  "-0.0900316-0.402928i -0.0688947-0.384273i -0.00546883-0.338719i 0.0908951-0.27376i 0.204577-0.199272i "
  "0.316122-0.125473i 0.405457-0.0609211i 0.455338-0.0110414i 0.454374+0.0214361i 0.399082+0.0442039i "
  "0.294553+0.0558001i 0.153625+0.0595931i -0.00530516+0.0583568i -0.161409+0.0534399i -0.294901+0.0445146i "
  "-0.39021+0.0299274i -0.439369+0.00750264i -0.439204-0.024472i -0.394172-0.0663899i -0.317769-0.116857i "
  "-0.225798-0.172746i -0.134088-0.229751i -0.0554523-0.283267i 0.00238914-0.329341i 0.0375132-0.366486i "
  "0.053075-0.38374i 0.0554523-0.394172i 0.0518088-0.400048i 0.0477465-0.40385i 0.0456556-0.406202i 0.0441668-0.405457i "
  "0.0387985-0.398017i 0.0235797-0.38042i -0.0110414-0.342772i -0.0609211-0.288736i -0.125473-0.221136i "
  "-0.199272-0.146221i -0.27376-0.0728483i -0.338719-0.0109377i -0.384273+0.0304597i -0.402928+0.0460876i "
  "-0.386975+0.027757i -0.344536-0.0167543i -0.280729-0.079817i -0.204577-0.151526i -0.126288-0.221951i "
  "-0.0554523-0.283267i 0.000532387-0.331198i 0.0375132-0.366486i 0.0549318-0.381884i 0.0554523-0.394172i "
  "0.0440088-0.407848i 0.0265258-0.42507i 0.00861562-0.443242i -0.00546883-0.455093i -0.0134306-0.450246i "
  "-0.0150053-0.419005i -0.0115738-0.343304i -0.00546883-0.233284i 0.000815537-0.0948476i 0.00530516+0.0583568i "
  "0.00696875+0.207881i 0.00581667+0.333598i 0.0027027+0.417435i -0.00214361+0.446872i 0.0027027+0.417435i "
  "0.00581667+0.333598i 0.00696875+0.207881i 0.00530516+0.0583568i 0.000815537-0.0948476i -0.00546883-0.233284i "
  "-0.0115738-0.343304i -0.0139335-0.417933i -0.0161333-0.452949i -0.0112855-0.460909i 0.00164687-0.45021i "
  "0.0212207-0.430376i 0.0431932-0.408664i 0.0609211-0.388703i 0.0665056-0.37031i 0.0525185-0.349337i "
  "0.013963-0.323172i -0.0499834-0.289432i -0.134904-0.244504i -0.231103-0.188662i -0.324738-0.125457i "
  "-0.399988-0.0612689i -0.441907-0.00402706i -0.439369+0.0375132i -0.387508+0.0594912i -0.289084+0.0612689i "
  "-0.154441+0.0431774i 0+0.0106103i 0.154441-0.0276089i 0.289084-0.0605732i 0.387508-0.0772335i 0.439369-0.0696673i "
  "0.441907-0.0228341i 0.399988+0.0503313i 0.324738+0.142688i 0.231103+0.241714i 0.134904+0.332522i "
  "0.0499834+0.400336i -0.013963+0.433036i -0.0525185+0.424364i -0.0665056+0.371374i -0.0609211+0.277798i "
  "-0.0431932+0.156088i -0.0212207+0.0212207i -0.00164687-0.111247i 0.0112855-0.228163i 0.0161333-0.321002i "
  "0.0128617-0.386851i 0.0142765-0.427944i 0.0112855-0.449972i 0.00615321-0.459641i 0-0.462207i -0.00615321-0.459641i "
  "-0.0112855-0.449972i -0.0142765-0.427944i -0.0150053-0.386851i -0.0107279-0.321002i 0.000347842-0.228163i "
  "0.0155844-0.111247i 0.031831+0.0212207i 0.0448243+0.156088i 0.0499834+0.277798i 0.0433579+0.371374i";
  assert_cvec(ref_c, csamples);
  ref_c = "1.14877e-06+0i -5.79353e-06+0i -5.16411e-06+0i 1.65031e-05+0i 6.01302e-05+0i 0.000107998+0i 0.000126412+0i "
  "7.86767e-05+0i -5.4505e-05-1.14877e-06i -0.00025453+5.79353e-06i -0.000454107+5.16411e-06i -0.00054513-1.65031e-05i "
  "-0.00040928-6.01302e-05i 3.26645e-05-0.000107998i 0.000765824-0.000126412i 0.00164039-7.86767e-05i 0.00236749+5.4505e-05i "
  "0.00255337+0.00025453i 0.00183369+0.000454107i -1.72249e-05+0.00054513i -0.00295681+0.00040928i -0.00658312-3.26645e-05i "
  "-0.0101241-0.000765824i -0.0125382-0.00164039i -0.0127351-0.00236634i -0.00987681-0.00255917i -0.0037585-0.00183885i "
  "0.00492827+3.37281e-05i 0.0144692+0.00301694i 0.0222362+0.00669112i 0.025074+0.0102505i 0.0199476+0.0126168i "
  "0.00476824+0.0126783i -0.0207501+0.00963387i -0.0543925+0.00331472i -0.0911003-0.0055064i -0.123147-0.0149987i "
  "-0.140917-0.0224196i -0.134256-0.024561i -0.0942497-0.0184646i -0.0151453-0.00229404i 0.103923+0.0238241i "
  "0.257782+0.0571447i 0.434953+0.0921403i 0.618478+0.120889i 0.787837+0.134052i 0.921774+0.122347i 1.00163+0.0782734i "
  "1.01466-0.00221228i 0.956699-0.118415i 0.833718-0.264315i 0.661716-0.428851i 0.464984-0.597096i 0.272817-0.752176i "
  "0.11513-0.877604i 0.0176536-0.959653i -0.002492-0.989322i 0.0599364-0.963554i 0.19683-0.885613i 0.387394-0.764307i "
  "0.600932-0.612504i 0.801383-0.445158i 0.952877-0.277241i 1.02542-0.121958i 0.999715+0.0104849i 0.870345+0.113445i "
  "0.646754+0.183694i 0.35175+0.220843i 0.0179555+0.226489i -0.317224+0.203352i -0.617217+0.154598i -0.85153+0.0834514i "
  "-0.999734-0.00690592i -1.05361-0.113243i -1.01727-0.23207i -0.905208-0.359534i -0.739017-0.491435i -0.543387-0.623365i "
  "-0.342212-0.750929i -0.155501-0.869953i 0.0024659-0.976598i 0.123583-1.0673i 0.205125-1.13875i 0.248149-1.18759i "
  "0.255608-1.21042i 0.230749-1.20393i 0.176161-1.1654i 0.0936737-1.09351i -0.0149818-0.989149i -0.147035-0.856266i "
  "-0.297489-0.702322i -0.458324-0.538057i -0.618418-0.376641i -0.764341-0.232223i -0.881946-0.118129i -0.958448-0.0450215i "
  "-0.984588-0.0194324i -0.956366-0.0429516i -0.876067-0.112261i -0.752127-0.219976i -0.597975-0.356077i -0.430049-0.509566i "
  "-0.26535-0.66993i -0.119003-0.828077i -0.00226564-0.976543i 0.0786795-1.109i 0.123049-1.21942i 0.13483-1.30095i "
  "0.121478-1.34543i 0.0922487-1.34353i 0.056495-1.28597i 0.0222508-1.16543i -0.00471259-0.978988i -0.0212782-0.730363i "
  "-0.0268539-0.431444i -0.022931-0.102415i -0.0123911+0.229627i 0.00121738+0.533584i 0.014331+0.77839i 0.0238911+0.937463i "
  "0.027724+0.992771i 0.0248041+0.937721i 0.0153989+0.778308i 0.00116749+0.532195i -0.0149386+0.22602i -0.0289689-0.108637i "
  "-0.0365239-0.439825i -0.0335619-0.739512i -0.0173931-0.986829i 0.0122953-1.16976i 0.0526101-1.28527i 0.097069-1.33769i "
  "0.135887-1.33623i 0.15705-1.29218i 0.148128-1.21637i 0.0986329-1.11756i 0.00250145-1.00187i -0.139753-0.873213i "
  "-0.319742-0.734629i -0.521149-0.589586i -0.721122-0.443193i -0.893044-0.302671i -1.01032-0.176894i -1.05062-0.0750385i "
  "-0.999736-0.00463313i -0.854513+0.0305213i -0.624153+0.0319934i -0.329421+0.00743645i -6.01302e-05-0.0299373i "
  "0.329297-0.0631639i 0.624032-0.0743647i 0.85444-0.0481098i 0.999787+0.0253505i 1.05088+0.147654i 1.01079+0.31122i "
  "0.893556+0.499822i 0.721411+0.690715i 0.520901+0.858007i 0.318724+0.976665i 0.137955+1.02644i";
  assert_cvec(ref_c, crsymbols);

  csamples = crc_tx.shape_symbols(csymbols);

  // Raised Cosine; complex input
  ref = "8.95103e-34 0.00113767 0.00477279 0.0105661 0.0171489 0.0221856 0.0227496 0.015992 -1.29939e-17 -0.0253289 "
  "-0.0576127 -0.0917169 -0.120042 -0.133416 -0.122502 -0.0795251 3.06162e-17 0.115879 0.262504 0.428983 0.600211 "
  "0.758749 0.887236 0.970942 1 0.970942 0.887236 0.758749 0.600211 0.428983 0.262504 0.115879 3.06162e-17 -0.0795251 "
  "-0.122502 -0.133416 -0.120042 -0.0917169 -0.0576127 -0.0253289 -1.29939e-17 0.015992 0.0227496 0.0221856 0.0171489 "
  "0.0105661 0.00477279 0.00113767 8.95103e-34";
  assert_vec(ref, crc_tx.get_pulse_shape());
  ref_c = "8.95103e-34+0i 0.00113767+0i 0.00477279+0i 0.0105661+0i 0.0171489+0i 0.0221856+0i 0.0227496+0i 0.015992+0i "
  "-1.29939e-17-8.95103e-34i -0.0253289-0.00113767i -0.0576127-0.00477279i -0.0917169-0.0105661i -0.120042-0.0171489i "
  "-0.133416-0.0221856i -0.122502-0.0227496i -0.0795251-0.015992i 3.06162e-17+1.29939e-17i 0.117017+0.0253289i "
  "0.267277+0.0576127i 0.439549+0.0917169i 0.61736+0.120042i 0.780934+0.133416i 0.909986+0.122502i 0.986934+0.0795251i "
  "1-3.06162e-17i 0.944475-0.115879i 0.824851-0.262504i 0.656466-0.428983i 0.46302-0.600211i 0.273381-0.758749i "
  "0.117252-0.887236i 0.0203623-0.970942i 7.42262e-17-1i 0.0616832-0.972079i 0.197615-0.892009i 0.387284-0.769315i "
  "0.600211-0.61736i 0.800447-0.451168i 0.952125-0.285253i 1.02514-0.131871i 1-1.76223e-17i 0.871054+0.103716i "
  "0.647482+0.175342i 0.351951+0.214566i 0.0171489+0.222935i -0.3192+0.202947i -0.61996+0.157365i -0.853925+0.088862i "
  "-1-4.62836e-18i -1.0516-0.106542i -1.01451-0.227641i -0.90273-0.359451i -0.737402-0.497318i -0.542885-0.635899i "
  "-0.342866-0.769507i -0.1572-0.892554i -3.06162e-17-1i 0.120846-1.08796i 0.202864-1.15451i 0.247318-1.1983i "
  "0.257233-1.21757i 0.235699-1.20992i 0.184887-1.17249i 0.105992-1.10281i -1.76223e-17-1i -0.131871-0.867225i "
  "-0.285253-0.711894i -0.451168-0.544182i -0.61736-0.377275i -0.769315-0.226036i -0.892009-0.105139i "
  "-0.972079-0.0270174i -1-3.52445e-17i -0.970942-0.0258797i -0.887236-0.100366i -0.758749-0.21547i "
  "-0.600211-0.360127i -0.428983-0.521996i -0.262504-0.689145i -0.115879-0.851233i -3.06162e-17-1i 0.0795251-1.12928i "
  "0.122502-1.23487i 0.133416-1.3122i 0.120042-1.35476i 0.0917169-1.3539i 0.0576127-1.29976i 0.0253289-1.18348i "
  "1.29939e-17-1i -0.015992-0.751346i -0.0227496-0.449391i -0.0221856-0.115199i -0.0171489+0.222935i "
  "-0.0105661+0.532713i -0.00477279+0.782097i -0.00113767+0.943924i -1.79021e-33+1i -0.00113767+0.943924i "
  "-0.00477279+0.782097i -0.0105661+0.532713i -0.0171489+0.222935i -0.0221856-0.115199i -0.0227496-0.449391i "
  "-0.015992-0.751346i 1.29939e-17-1i 0.0264666-1.18234i 0.0623855-1.29499i 0.102283-1.34333i 0.137191-1.33761i "
  "0.155601-1.29001i 0.145251-1.21213i 0.095517-1.11329i -4.36101e-17-1i -0.141208-0.874287i -0.320116-0.737212i "
  "-0.5207-0.592581i -0.720253-0.445871i -0.892164-0.304514i -1.00974-0.177369i -1.05047-0.0734208i -1-3.06162e-17i "
  "-0.855062+0.0370665i -0.624732+0.0373666i -0.329766+0.00894711i 0-0.0342978i 0.329766-0.0744505i "
  "0.624732-0.0924114i 0.855062-0.0713258i 1+5.6604e-17i 1.05047+0.124079i 1.00974+0.292594i 0.892164+0.487948i "
  "0.720253+0.685955i 0.5207+0.859413i 0.320116+0.982215i 0.141208+1.03334i 4.36101e-17+1i -0.095517+0.881529i "
  "-0.145251+0.687118i -0.155601+0.432049i -0.137191+0.137191i -0.102283-0.174165i -0.0623855-0.479481i "
  "-0.0264666-0.759545i -1.29939e-17-1i 0.0148543-1.19168i 0.0179768-1.32985i 0.0116196-1.41286i 0-1.44051i "
  "-0.0116196-1.41286i -0.0179768-1.32985i -0.0148543-1.19168i 1.29939e-17-1i 0.0241912-0.759545i 0.0528399-0.479481i "
  "0.0811508-0.174165i 0.102893+0.137191i 0.11123+0.432049i 0.0997521+0.687118i 0.0635331+0.881529i";
  assert_cvec(ref_c, csamples);
}
