package org.jboss.cache.factories.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Used to annotate a method as one that is used to inject a registered component into another component.  The component
 * to be constructed must be built using the {@link org.jboss.cache.factories.ComponentFactory#construct(Class)} method, or if your object that needs
 * components injected into it already exists, it can be built using the {@link org.jboss.cache.factories.ComponentFactory#wireComponents(Object)}
 * method.
 * <p/>
 * Usage example:
 * <pre>
 *       public class MyClass
 *       {
 *          private TransactionManager tm;
 *          private BuddyManager bm;
 *          private Notifier n;
 * <p/>
 *          &amp;Inject
 *          public void setTransactionManager(TransactionManager tm)
 *          {
 *             this.tm = tm;
 *          }
 * <p/>
 *          &amp;Inject
 *          public void injectMoreStuff(BuddyManager bm, Notifier n)
 *          {
 *             this.bm = bm;
 *             this.n = n;
 *          }
 *       }
 * <p/>
 * </pre>
 * and an instance of this class can be constructed and wired using
 * <pre>
 *       MyClass myClass = componentFactory.construct(MyClass.class); // instance will have dependencies injected.
 * </pre>
 * If you wish to use named components, you can use the optional {@link org.jboss.cache.factories.annotations.ComponentName}
 * annotation on each parameter.
 *
 * @author Manik Surtani
 * @see ComponentName
 * @since 2.1.0
 */
// ensure this annotation is available at runtime.
@Retention(RetentionPolicy.RUNTIME)

// only applies to fields.
@Target(ElementType.METHOD)
public @interface Inject
{
}
