package org.jboss.cache.lock;

import org.jboss.cache.Fqn;

import java.util.Collection;
import java.util.Set;


/**
 * Interface for a lock for nodes.
 */
public interface NodeLock
{

   Fqn getFqn();

   /**
    * Returns a copy of the reader lock owner in List.
    * Size is zero is not available. Note that this list
    * is synchronized.
    *
    * @return Collection<Object> of readers
    */
   Collection<Object> getReaderOwners();

   /**
    * Returns the writer lock owner object. Null if not available.
    *
    * @return Object owner
    */
   Object getWriterOwner();

   /**
    * Acquires a write lock with a timeout of <code>timeout</code> milliseconds.
    * Note that if the current owner owns a read lock, it will be upgraded
    * automatically. However, if upgrade fails, i.e., timeout, the read lock will
    * be released automatically.
    *
    * @param caller  Can't be null.
    * @param timeout
    * @return boolean True if lock was acquired and was not held before, false if lock was held
    * @throws LockingException
    * @throws TimeoutException
    */
   boolean acquireWriteLock(Object caller, long timeout) throws LockingException, TimeoutException,
         InterruptedException;

   /**
    * Acquires a read lock with a timeout period of <code>timeout</code> milliseconds.
    *
    * @param caller  Can't be null.
    * @param timeout
    * @return boolean True if lock was acquired and was not held before, false if lock was held
    * @throws LockingException
    * @throws TimeoutException
    */
   boolean acquireReadLock(Object caller, long timeout) throws LockingException, TimeoutException, InterruptedException;

   /**
    * Releases the lock held by the owner.
    *
    * @param caller Can't be null.
    */
   void release(Object caller);

   /**
    * Releases all locks associated with this instance.
    */
   void releaseAll();

   /**
    * Releases all locks with this owner.
    */
   void releaseAll(Object owner);

   /**
    * Check if there is a read lock.
    */
   boolean isReadLocked();

   /**
    * Check if there is a write lock.
    */
   boolean isWriteLocked();

   /**
    * Check if there is a read or write lock
    */
   boolean isLocked();

   /**
    * Returns true if the object is the lock owner.
    */
   boolean isOwner(Object o);

   boolean acquire(Object caller, long timeout, LockType lock_type) throws LockingException, TimeoutException,
         InterruptedException;

   /**
    * Recursively acquire locks for this node and all subnodes, including internal Fqns such as buddy backup subtrees.
    *
    * @param caller    lock owner
    * @param timeout   time to wait
    * @param lock_type type of lock
    * @return locks acquired
    */
   Set<NodeLock> acquireAll(Object caller, long timeout, LockType lock_type) throws LockingException, TimeoutException,
         InterruptedException;

   /**
    * Same as the overloaded {@link #acquire(Object, long, LockType)} except that you can
    * optionally specify that internal Fqns - such as buddy backup subtrees - can be excluded when acquiring locks.
    *
    * @param caller              lock owner
    * @param timeout             time to wait
    * @param lock_type           type of lock
    * @param excludeInternalFqns if true, locks on internal fqns are not acquired.
    * @return locks acquired
    */
   Set<NodeLock> acquireAll(Object caller, long timeout, LockType lock_type, boolean excludeInternalFqns) throws LockingException, TimeoutException,
         InterruptedException;

   void printLockInfo(StringBuilder sb, int indent);

}