/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.xml;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.util.StringPropertyReplacer;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

/**
 * A simple XML utility class for reading configuration elements
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 */
public class XmlHelper
{
   private static final Log log = LogFactory.getLog(XmlHelper.class);

   /**
    * The root of a JBoss Cache configuration XML file.  This is the <tt>&lt;mbean&gt;</tt> tag.
    */
   public static final String ROOT = "mbean";

   /**
    * The <tt>&lt;attribute&gt;</tt> tag which forms the bulk of JBoss Cache configuration elements
    */
   public static final String ATTR = "attribute";

   /**
    * The <tt>&lt;config&gt;</tt> tag may be embedded in the contents of an <tt>&lt;attribute&gt;</tt>, to specify more
    * complex configuration for certain parameters.
    */
   public static final String CONFIG_ATTR = "config";

   /**
    * The <tt>&lt;name&gt;</tt> attribute to an <tt>&lt;attribute&gt;</tt> tag.
    */
   public static final String NAME = "name";


   /**
    * Returns the contents of a specific node of given element name, provided a certain attribute exists and is set to value.
    * E.g., if you have a {@link Element} which represents the following XML snippet:
    * <pre>
    *   &lt;ItemQuantity Colour="Red"&gt;100&lt;/ItemQuantity&gt;
    *   &lt;ItemQuantity Colour="Blue"&gt;30&lt;/ItemQuantity&gt;
    *   &lt;ItemQuantity Colour="Black"&gt;10&lt;/ItemQuantity&gt;
    * <pre>
    * <p/>
    * The following results could be expected:
    * </p>
    * <pre>
    *    getTagContents(element, "Red", "ItemQuantity", "Colour"); // 100
    *    getTagContents(element, "Black", "ItemQuantity", "Colour"); // 10
    *    getTagContents(element, "Blah", "ItemQuantity", "Colour"); // null
    *    getTagContents(element, "Red", "Blah", "Colour"); // null
    *    getTagContents(element, "Black", "ItemQuantity", "Blah"); // null
    * </pre>
    * <p/>
    * None of the parameters should be null - otherwise the method may throw a NullPointerException.
    * </p>
    *
    * @param elem          - element to search through.
    * @param value         - expected value to match against
    * @param elementName   - element name
    * @param attributeName - attribute name of the element that would contain the expected value.
    * @return the contents of the matched element, or null if not found/matched
    */
   public static String getTagContents(Element elem, String value, String elementName, String attributeName)
   {
      NodeList list = elem.getElementsByTagName(elementName);

      for (int s = 0; s < list.getLength(); s++)
      {
         org.w3c.dom.Node node = list.item(s);
         if (node.getNodeType() != org.w3c.dom.Node.ELEMENT_NODE)
            continue;

         Element element = (Element) node;
         String name = element.getAttribute(attributeName);
         if (name.equals(value))
         {
            return getElementContent(element, true);
         }
      }
      return null;
   }

   /**
    * Retrieves the value of a given attribute for the first encountered instance of a tag in an element.
    * <p/>
    * E.g., if you have a {@link Element} which represents the following XML snippet:
    * </p>
    * <pre>
    *   &lt;ItemQuantity Colour="Red"&gt;100&lt;/ItemQuantity&gt;
    *   &lt;ItemQuantity Colour="Blue"&gt;30&lt;/ItemQuantity&gt;
    *   &lt;ItemQuantity Colour="Black"&gt;10&lt;/ItemQuantity&gt;
    * <pre>
    * <p/>
    * The following results could be expected:
    * </p>
    * <pre>
    *    getAttributeValue(element, "ItemQuantity", "Colour"); // "Red"
    *    getTagContents(element, "Blah", "Colour"); // null
    *    getTagContents(element, "ItemQuantity", "Blah"); // null
    * </pre>
    * None of the parameters should be null - otherwise the method may throw a NullPointerException.
    *
    * @param elem          - element to search through.
    * @param elementName   - element name
    * @param attributeName - attribute name of the element that would contain the expected value.
    * @return the contents of the matched attribute, or null if not found/matched
    */
   public static String getAttributeValue(Element elem, String elementName, String attributeName)
   {
      NodeList list = elem.getElementsByTagName(elementName);

      for (int s = 0; s < list.getLength(); s++)
      {
         org.w3c.dom.Node node = list.item(s);
         if (node.getNodeType() != org.w3c.dom.Node.ELEMENT_NODE)
            continue;

         Element element = (Element) node;
         String value = element.getAttribute(attributeName);
         return value == null ? null : StringPropertyReplacer.replaceProperties(value);

      }
      return null;
   }

   /**
    * Convenience method, equivalent to calling <tt>getSubElement(element, "config");</tt>
    */
   public static Element getConfigSubElement(Element element)
   {
      return getSubElement(element, CONFIG_ATTR);
   }

   /**
    * Returns a named sub-element of the current element passed in.
    * <p/>
    * None of the parameters should be null - otherwise the method may throw a NullPointerException.
    *
    * @param element        - element to search through.
    * @param subElementName - the name of a sub element to look for
    * @return the first matching sub element, if found, or null otherwise.
    */
   public static Element getSubElement(Element element, String subElementName)
   {
      NodeList nl = element.getChildNodes();
      for (int i = 0; i < nl.getLength(); i++)
      {
         Node node = nl.item(i);
         if (node.getNodeType() == Node.ELEMENT_NODE && subElementName.equals(((Element) node).getTagName()))
         {
            return (Element) node;
         }
      }

      if (log.isDebugEnabled()) log.debug("getSubElement(): Does not exist for " + subElementName);
      return null;
   }

   /**
    * Reads the contents of the element passed in.
    * <p/>
    * None of the parameters should be null - otherwise the method may throw a NullPointerException.
    *
    * @param element - element to search through.
    * @param trim    - if true, whitespace is trimmed before returning
    * @return the contents of the element passed in.  Will return an empty String if the element is empty.
    */
   public static String getElementContent(Element element, boolean trim)
   {
      NodeList nl = element.getChildNodes();
      String attributeText = "";
      for (int i = 0; i < nl.getLength(); i++)
      {
         Node n = nl.item(i);
         if (n instanceof Text)
         {
            attributeText += StringPropertyReplacer.replaceProperties(((Text) n).getData());
         }
      } // end of for ()
      if (trim)
         attributeText = attributeText.trim();
      return attributeText;
   }

   /**
    * Reads the contents of the first occurence of elementName under the given element, trimming results of whitespace.
    * <p/>
    * None of the parameters should be null - otherwise the method may throw a NullPointerException.
    *
    * @param element     - element to search through.
    * @param elementName - name of the element to find within the element passed in
    * @return may return an empty String of not found.
    */
   public static String readStringContents(Element element, String elementName)
   {
      NodeList nodes = element.getElementsByTagName(elementName);
      if (nodes.getLength() > 0)
      {
         Node node = nodes.item(0);
         Element ne = (Element) node;
         NodeList nl2 = ne.getChildNodes();
         Node node2 = nl2.item(0);
         if (node2 != null)
         {
            String value = node2.getNodeValue();
            if (value == null)
               return "";
            return StringPropertyReplacer.replaceProperties(value.trim());
         }
         else
         {
            return "";
         }
      }
      else
      {
         return "";
      }
   }

   /**
    * Escapes backslashes ('\') with additional backslashes in a given String, returning a new, escaped String.
    *
    * @param value String to escape.   Cannot be null.
    * @return escaped String.  Never is null.
    */
   public static String escapeBackslashes(String value)
   {
      StringBuilder buf = new StringBuilder(value);
      for (int looper = 0; looper < buf.length(); looper++)
      {
         char curr = buf.charAt(looper);
         char next = 0;
         if (looper + 1 < buf.length())
            next = buf.charAt(looper + 1);

         if (curr == '\\')
         {
            if (next != '\\')
            {           // only if not already escaped
               buf.insert(looper, '\\');  // escape backslash
            }
            looper++;                    // skip past extra backslash (either the one we added or existing)
         }
      }
      return buf.toString();
   }

   /**
    * Reads the contents of a named sub element within a given element, and attempts to parse the contents as a Java
    * properties file.
    * <p/>
    * E.g., if you have a {@link Element} which represents the following XML snippet:
    * <p/>
    * <pre>
    *   &lt;props&gt;
    *       my.attrib.1 = blah
    *       my.attrib.2 = blahblah
    *   &lt;/props&gt;
    * <pre>
    * <p/>
    * The following results could be expected:
    * <p/>
    * <pre>
    *    Properties p = readPropertiesContents(element, "props");
    *    p.getProperty("my.attrib.1"); // blah
    *    p.getProperty("my.attrib.2"); // blahblah
    * </pre>
    * None of the parameters should be null - otherwise the method may throw a NullPointerException.
    *
    * @param element     - element to search through.
    * @param elementName - name of the element to find within the element passed in
    * @return a {@link Properties} object, never null.
    * @throws IOException if unable to parse the contents of the element
    */
   public static Properties readPropertiesContents(Element element, String elementName) throws IOException
   {
      String stringContents = readStringContents(element, elementName);
      if (stringContents == null) return new Properties();
      // JBCACHE-531: escape all backslash characters
      stringContents = escapeBackslashes(stringContents);
      ByteArrayInputStream is = new ByteArrayInputStream(stringContents.trim().getBytes("ISO8859_1"));
      Properties properties = new Properties();
      properties.load(is);
      is.close();
      return properties;
   }

   /**
    * Similar to {@link #readStringContents(org.w3c.dom.Element,String)} except that it returns a boolean.
    *
    * @param element     - element to search through.
    * @param elementName - name of the element to find within the element passed in
    * @return the contents of the element as a boolean, or false if not found.
    */
   public static boolean readBooleanContents(Element element, String elementName)
   {
      return readBooleanContents(element, elementName, false);
   }

   /**
    * Similar to {@link #readStringContents(org.w3c.dom.Element,String)} except that it returns a boolean.
    *
    * @param element      - element to search through.
    * @param elementName  - name of the element to find within the element passed in
    * @param defaultValue - value to return if the element is not found or cannot be parsed.
    * @return the contents of the element as a boolean
    */
   public static boolean readBooleanContents(Element element, String elementName, boolean defaultValue)
   {
      String val = readStringContents(element, elementName);
      if (val.equalsIgnoreCase("true") || val.equalsIgnoreCase("false"))
      {
         // needs to be done this way because of JBBUILD-351
         return Boolean.valueOf(val);
         //return Boolean.parseBoolean(val);
      }
      return defaultValue;
   }

   /**
    * Converts a String representing an XML snippet into an {@link org.w3c.dom.Element}.
    *
    * @param xml snippet as a string
    * @return a DOM Element
    * @throws Exception if unable to parse the String or if it doesn't contain valid XML.
    */
   public static Element stringToElement(String xml) throws Exception
   {
      ByteArrayInputStream bais = new ByteArrayInputStream(xml.getBytes("utf8"));
      DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
      Document d = builder.parse(bais);
      bais.close();
      return d.getDocumentElement();
   }

   /**
    * Returns the root element of a given input stream
    *
    * @param is stream to parse
    * @return XML DOM element, or null if unable to parse stream
    */
   public static Element getDocumentRoot(InputStream is)
   {
      Document doc;
      try
      {
         InputSource xmlInp = new InputSource(is);

         DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
         DocumentBuilder parser = docBuilderFactory.newDocumentBuilder();
         doc = parser.parse(xmlInp);
         Element root = doc.getDocumentElement();
         root.normalize();
         return root;
      }
      catch (SAXParseException err)
      {
         log.error("Configurator SAXParse error", err);
      }
      catch (SAXException e)
      {
         log.error("Configurator SAX error", e);
      }
      catch (Exception pce)
      {
         log.error("Configurator general error", pce);
      }
      return null;
   }

   /**
    * Retrieves the boolean value of a given attribute for the first encountered instance of elementName
    *
    * @param elem          - element to search
    * @param elementName   - name of element to find
    * @param attributeName - name of attribute to retrieve the value of
    * @param defaultValue  - default value to return if not found
    */
   public static boolean readBooleanAttribute(Element elem, String elementName, String attributeName, boolean defaultValue)
   {
      String val = getAttributeValue(elem, elementName, attributeName);
      if (val != null && (val.equalsIgnoreCase("true") || val.equalsIgnoreCase("false")))
      {
         //return Boolean.parseBoolean(val);
         // needs to be done this way because of JBBUILD-351
         return Boolean.valueOf(val);
      }

      return defaultValue;
   }

}
