/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.spi.editor.hints;

import java.util.ArrayList;
import java.util.List;
import javax.swing.text.Position;
import org.openide.filesystems.FileObject;

/**
 * Represents a set of changes made by a hint.  Note that the start/end
 * parameters refer to offsets that should be <i>selected</i>, not to
 * the general offsets of the modified element.
 * <p>
 * In other words, a change generated by a hint can generate things like
 * variable names;  ChangeInfo provides a list of generated code which the
 * user might want to modify/replace with their own text.  The start/end offsets
 * are offsets into the file in question, which determine selection start/end
 * in the editor.
 * <p>
 * This class provides for a list of changes, anticipating &quot;live template&quot;
 * support (where the user confirms each generated element).  Currently only
 * the first change provided is used.
 * 
 * @author Tim Boudreau
 */
public final class ChangeInfo {
    private List<Change> changes = null;
    
    /**
     * Create an instance of ChangeInfo prepopulated with a
     * single change
     */
    public ChangeInfo(FileObject fileObject, Position start, Position end) {
        add (fileObject, start, end);
    }
    
    /**
     * Create an instance of ChangeInfo prepopulated with a
     * single change without associated file object.
     */
    public ChangeInfo(Position start, Position end) {
        add (null, start, end);
    }
    
    /**
     * Create an empty instance of ChangeInfo .
     */
    public ChangeInfo() {
    }

    /**
     * @return number of changes
     */
    public final int size() {
        return changes != null ? changes.size() : 0;
    }

    /**
     * Adds one change.
     * @param fileObject
     * @param start
     * @param end
     */
    public final void add (FileObject fileObject, Position start, Position end) {
        if (changes == null) {
            changes = new ArrayList<Change>(5);
        }
        changes.add (new ChangeImpl (fileObject, start, end));
    }

    /**
     * Getter for the i-th change
     * @param i index of the change
     * @return the change or throws ArrayIndexOutOfBoundsException if there are
     *         no changes in the ChangeInfo
     */
    public final Change get (int i) {
        if (changes == null) {
            throw new ArrayIndexOutOfBoundsException ("No changes");
        }
        return (Change) changes.get(i);
    }
    
    @Override
    public String toString() {
        int size = size();
        if (size == 0) {
            return "Empty ChangeInfo";
        } else {
            StringBuffer sb = new StringBuffer (100);
            sb.append ("ChangeInfo [");
            for (int i=0; i < size; i++) {
                sb.append (get(i));
                if (i != size-1) {
                    sb.append (',');
                }
            }
            sb.append ("]");
            return sb.toString();
        }
    }
    
    /**
     * Interface representing a single caret-positioning or user-modifiable
     * change.
     */
    public static interface Change {
        /**
         * @return start position of the change
         */
        public Position getStart();

        /**
         * @return end position of the change
         */
        public Position getEnd();
        /**
         * @return the associated file or <code>null</code> if there is no
         *         file associated with the chage
         */
        public FileObject getFileObject();
    }
    
    private static final class ChangeImpl implements Change {
        Position start;
        Position end;
        FileObject fileObject;
        
        ChangeImpl(FileObject fileObject, Position start, Position end) {
            this.fileObject = fileObject;
            this.start = start;
            this.end = end;
        }
        
        public Position getStart() {
            return start;
        }
        
        public Position getEnd() {
            return end;
        }
        
        public FileObject getFileObject() {
            return fileObject;
        }
        
        public String toString() {
            return "Change from " + start.getOffset() + " to " + end.getOffset() + " in " + fileObject;
        }
    }
    
}
