/* Copyright 2023 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#include "tensorflow/compiler/xla/python/ifrt/support/sharding_param_to_op_sharding.h"

#include <cstdint>

#include "absl/types/span.h"
#include "llvm/ADT/SmallVector.h"
#include "tensorflow/compiler/xla/python/ifrt/ir/sharding_param.h"
#include "tensorflow/compiler/xla/statusor.h"
#include "tensorflow/compiler/xla/xla_data.pb.h"
#include "tensorflow/tsl/platform/errors.h"

namespace xla {
namespace ifrt {
namespace support {

StatusOr<OpSharding> ToOpSharding(const ShardingParam& sharding_param,
                                  absl::Span<const int> device_mapping) {
  OpSharding op_sharding;
  op_sharding.set_type(OpSharding::OTHER);

  // Populate tile_assignment_dimensions.
  auto* tile_assignment_dims = op_sharding.mutable_tile_assignment_dimensions();
  int64_t cum_size = 1;
  tile_assignment_dims->Reserve(sharding_param.dim_shards().size() + 1);
  for (const int64_t dim_shard : sharding_param.dim_shards()) {
    cum_size *= dim_shard;
    tile_assignment_dims->Add(dim_shard);
  }
  int device_count = 1;
  for (const int axis_size : sharding_param.minor_to_major().axis_sizes) {
    device_count *= axis_size;
  }
  if (device_count != cum_size) {
    op_sharding.set_replicate_on_last_tile_dim(true);
    tile_assignment_dims->Add(device_count / cum_size);
  }

  // Populate tile_assignment_devices.
  llvm::SmallVector<int, 4> devices;
  sharding_param.minor_to_major().ToDeviceList(devices);
  auto* tile_assignment_devices = op_sharding.mutable_tile_assignment_devices();
  tile_assignment_devices->Reserve(devices.size());
  for (const int device : devices) {
    if (device < 0 || device >= device_mapping.size()) {
      return tsl::errors::OutOfRange("Can't map device ", device);
    }
    tile_assignment_devices->Add(device_mapping[device]);
  }

  return op_sharding;
}

}  // namespace support
}  // namespace ifrt
}  // namespace xla
